package com.l2client.controller;

import java.util.logging.Logger;

import com.jme.scene.Node;

/**
 * The scene manager provides means to access to the visual hierarchy, for
 * attaching terrain tiles (used by the @see SimpleTerrainManager), or as a
 * general access to the scene root (for attachment of variable kind) All visual
 * elements should be attached to the scene root in a somewhat consistent way.
 * 
 * This class is a singleton and should be accessed by using SceneManager.get()
 * 
 */
// TODO improve the handling of the terrain and root nodes, can cause
// inconsistences in case someone removes all nodes below root
public class SceneManager {

	private static Logger log = Logger.getLogger(SceneManager.class.getName());

	private static SceneManager singleton = null;
	/**
	 * Internal reference to the root node of the terrain tiles
	 */
	private volatile Node patches = null;
	/**
	 * 
	 */
	private volatile Node root = null;

	/**
	 * Internal constructor which also creates the terrain root, but does not
	 * attach it
	 */
	private SceneManager() {
		singleton = this;
		patches = new Node("Terrainpatches root");
	}

	/**
	 * Fetch the singleton instance (created in case not done so far)
	 * 
	 * @return The SceneManager instance
	 */
	public static SceneManager get() {
		if (singleton != null)
			return singleton;
		else {
			return new SceneManager();
		}
	}

	/**
	 * Sets the node of the scene to be used as visual root, anything rendered
	 * should be placed below. Terrain tiles should go to a special root, as
	 * they are swapped in and out dynamically.
	 * 
	 * @param n
	 *            node to be used as root (should be the one hooked up in the
	 *            rendering system)
	 */
	public void setRoot(Node n) {
		root = n;
		root.attachChild(patches);
	}

	/**
	 * Fetches the root node of the visual hierarchy also containing the terrain
	 * tiles, so be carefull.
	 * 
	 * @return The root node of the scene
	 */
	public Node getRoot() {
		return root;
	}

	/**
	 * Adds a complete visual representaion of a terrain tile to the scene for
	 * drawing
	 * 
	 * @param n
	 *            the root node tof the terain tile (can be a hierarchy by
	 *            itself)
	 */
	public void attachPatch(Node n) {
		if (root != null) {
			// TODO check this, is it really desired to always have the terrain
			// attached? ev. triggered by a new flag (indoor, or something like
			// this)
			if (patches.getParent() == null)
				root.attachChild(patches);

			patches.attachChild(n);
		} else
			log
					.severe("SceneManager attachPatch called without having a root (SceneManager not initialized?) "
							+ n.getName());
	}

	/**
	 * Removes the terrain tile from the scene (for example if moved out of
	 * view, and ready to unload)
	 * 
	 * @param n
	 *            the node to be removed
	 */
	public void detachPatch(Node n) {
		patches.detachChild(n);
	}
}
