package com.l2client.controller;

import java.util.concurrent.ConcurrentHashMap;

import javax.swing.ImageIcon;

import com.jme.bounding.OrientedBoundingBox;
import com.jme.image.Texture;
import com.jme.scene.Node;
import com.jme.scene.shape.Quad;
import com.jme.scene.state.CullState;
import com.jme.scene.state.TextureState;
import com.jme.system.DisplaySystem;
import com.jme.util.TextureManager;
import com.jmex.terrain.util.ProceduralTextureGenerator;
import com.l2client.model.SimpleHeightMap;

/**
 * A simple terrain manager, storing definitions of terrain tiles, loaded tiles, etc.
 * The simple terrain manager is used for testing only and does not use the @see AssetManager for loading.
 * The terrain model used is kept simple, just 128 sized plain quads are used to pose as terrain.
 * The swapping of terrain is already demonstrated, but in contrast to the real terrain manager the loading is handled synchronously.
 * As the character moves on new tiles are attached to the scene, as older are flushed from the cache.
 *
 * Used as a singleton by calling SimpleTerrainManager.get()
 */
public class SimpleTerrainManager {

	// private static int count = 99;
	// TerrainPatch load[] = new TerrainPatch[24];//7x7 -5x5 third ring low
	// detail in loading
	// TerrainPatch far[] = new TerrainPatch[16];//5x5 -3x3 second ring low
	// detail
	// TerrainPatch near[] = new TerrainPatch[9];//3x3 high detail
	/**
	 * Internal wrapper for a terrain patch, stores the coordinates of the
	 * patch, load information, detail information and after loading the finally
	 * loaded patch from the @see AssetManager
	 */
	//TODO getter/setter
	private class PatchInfo {
		public PatchInfo(int _x, int _y, boolean b) {
			x = _x;
			y = _y;
			detail = b;
		}

		public int x;
		public int y;
		public boolean detail;
		public Node patch = null;
	}
	
	/**
	 * The size of one terrain tile in x-y direction (z would be height)
	 */
	public static int TERRAIN_SIZE = 128;

	/**
	 *  Loadqueue 1: the to be unloaded patches after a move
	 */
	ConcurrentHashMap<Integer, PatchInfo> unloadPatches = new ConcurrentHashMap<Integer, PatchInfo>();
	/**
	 * Loadqueue 2: the to be loaded patches
	 */
	ConcurrentHashMap<Integer, PatchInfo> loadedPatches = new ConcurrentHashMap<Integer, PatchInfo>();
	/**
	 * internal reference to the center tile
	 */
	private PatchInfo center = null;
	/**
	 * internal singleton reference
	 */
	private static SimpleTerrainManager singleton = null;
	/**
	 * JME specific cull state to be used on the terrain tiles (only in simple) 
	 */
	private CullState cull = null;
	/**
	 * JME specific texture state to be used in rendering (only in simple) 
	 */
	private TextureState tex = null;

	/**
	 * internal singleton constructor, initializes dummy textures for the simple demo
	 */
	private SimpleTerrainManager() {
		singleton = this;
		initDummyTexture();
	}

	/**
	 * Fetch the singleton instance, creating one of it has not happened so far.
	 * @return
	 */
	public static SimpleTerrainManager get() {
		if (singleton != null)
			return singleton;
		else {
			return new SimpleTerrainManager();
		}
	}

	/**
	 * Sets the center coordinates. This will be used for calculation which tiles must be swapped. Early out if center has not changed.
	 * Expects the world coordinates/TERRAIN_SIZE as x and z
	 * 
	 * Should be called each render frame or at least once a second.
	 * 
	 * @param x	terrain coordinates in world coords/TERRAIN_SIZE
	 * @param y terrain coordinates in world coords/TERRAIN_SIZE
	 */
	public void setCenter(int x, int y) {

		if (center != null && center.x == x && center.y == y)
			return;

		unloadPatches.putAll(loadedPatches);
		for (PatchInfo p : loadedPatches.values())
			SceneManager.get().detachPatch(p.patch);

		loadedPatches.clear();

		initCenter(x, y);
		initFirstRing(x, y);
		initSecondRing(x, y);
		// initThirdRing(x,z);

		unloadPatches.clear();
		//JME specific to update scene
		SceneManager.get().getRoot().updateWorldBound();
		SceneManager.get().getRoot().updateRenderState();
	}

	/**
	 * initializes the second ring around the center tile
	 * @param x tile # (coords/tile_size) in x 
	 * @param z tile # (coords/tile_size) in y
	 */
	private void initSecondRing(int x, int z) {
		addLoadAll(checkLoadPatch(x - 2, z - 2));
		addLoadAll(checkLoadPatch(x - 1, z - 2));
		addLoadAll(checkLoadPatch(x, z - 2));
		addLoadAll(checkLoadPatch(x + 1, z - 2));
		addLoadAll(checkLoadPatch(x + 2, z - 2));
		addLoadAll(checkLoadPatch(x + 2, z - 1));
		addLoadAll(checkLoadPatch(x + 2, z));
		addLoadAll(checkLoadPatch(x + 2, z + 1));
		addLoadAll(checkLoadPatch(x + 2, z + 2));
		addLoadAll(checkLoadPatch(x + 1, z + 2));
		addLoadAll(checkLoadPatch(x, z + 2));
		addLoadAll(checkLoadPatch(x - 1, z + 2));
		addLoadAll(checkLoadPatch(x - 2, z + 2));
		addLoadAll(checkLoadPatch(x - 2, z + 1));
		addLoadAll(checkLoadPatch(x - 2, z));
		addLoadAll(checkLoadPatch(x - 2, z - 1));
	}
	
	/**
	 * initializes the first ring around the center tile
	 * @param x tile # (coords/tile_size) in x 
	 * @param z tile # (coords/tile_size) in y
	 */
	private void initFirstRing(int x, int z) {
		addLoadAll(checkLoadPatch(x - 1, z - 1));
		addLoadAll(checkLoadPatch(x, z - 1));
		addLoadAll(checkLoadPatch(x + 1, z - 1));
		addLoadAll(checkLoadPatch(x + 1, z));
		addLoadAll(checkLoadPatch(x + 1, z + 1));
		addLoadAll(checkLoadPatch(x, z + 1));
		addLoadAll(checkLoadPatch(x - 1, z + 1));
		addLoadAll(checkLoadPatch(x - 1, z));
	}

	/**
	 * start loading all and attach base + detail
	 * 
	 * @param p
	 * @return
	 */
	private PatchInfo addLoadAll(PatchInfo p) {
		loadedPatches.put(p.hashCode(), p);
		SceneManager.get().attachPatch(p.patch);
		// p.loadBase();
		// p.loadDetails();
		// //p.setDetail(true);
		// p.attach();
		return p;
	}

	/**
	 * initializes the center tile
	 * @param x tile # (coords/tile_size) in x 
	 * @param z tile # (coords/tile_size) in y
	 */
	private void initCenter(int x, int y) {
		center = addLoadAll(checkLoadPatch(x, y));
	}

	/**
	 * Checks if the to be loaded tile is present in the unloaded cache and eventually revives it,
	 * otherwise initializes loading of the tile in asynchronous mode via the @see AssetManager
	 *
	 * the simple version just creates quads synchronously
	 * 
	 * @param x tile # (coords/tile_size) in x 
	 * @param z tile # (coords/tile_size) in y
	 * @return a @see PatchInfo for the tile to be loaded (or not loaded in case tile not present)
	 */
	private PatchInfo checkLoadPatch(int x, int y) {
		PatchInfo ret = new PatchInfo(x, y, false);
		if (unloadPatches.contains(ret))
			return unloadPatches.remove(ret.hashCode());

		try {
			Quad q = new Quad("bottom", 1f * TERRAIN_SIZE, 1f * TERRAIN_SIZE);
			q.setModelBound(new OrientedBoundingBox());
			q.updateModelBound();
			Node n = new Node(x + " " + y);
			n.attachChild(q);
			n.setLocalTranslation(x * TERRAIN_SIZE, y * TERRAIN_SIZE, 0f);
			n.setRenderState(cull);
			n.setRenderState(tex);
			ret.patch = n;
		} catch (Exception e) {
			//TODO use logger & error handling on failed load of a tile (in which case is this ok?)
			e.printStackTrace();
		}

		return ret;
	}

	/**
	 * Initializes a dummy texture to be used by ALL tiles, this is just for the demo.
	 * Normally a tile already contains complete culling and rendering information, so this is not needed at all.
	 * JME specific
	 */
	private void initDummyTexture() {
		CullState cs = DisplaySystem.getDisplaySystem().getRenderer()
				.createCullState();
		cs.setCullFace(CullState.Face.Back);
		cs.setEnabled(true);

		SimpleHeightMap heightMap = new SimpleHeightMap(128, null);
		// Vector3f terrainScale = new Vector3f(5, 1, 5);
		// Vector3f terrainScale = new Vector3f(5, 1, 5);

		ProceduralTextureGenerator pt = new ProceduralTextureGenerator(
				heightMap);
		pt.addTexture(new ImageIcon(SimpleTerrainManager.class.getClassLoader()
				.getResource("textures/grassb.png")), -128, 0, 128);
		pt.addTexture(new ImageIcon(SimpleTerrainManager.class.getClassLoader()
				.getResource("textures/dirt.jpg")), 0, 128, 255);
		pt.addTexture(new ImageIcon(SimpleTerrainManager.class.getClassLoader()
				.getResource("textures/highest.jpg")), 128, 255, 384);

		pt.createTexture(512);

		TextureState ts = DisplaySystem.getDisplaySystem().getRenderer()
				.createTextureState();
		ts.setEnabled(true);

		Texture t1 = TextureManager.loadTexture(pt.getImageIcon().getImage(),
				Texture.MinificationFilter.Trilinear,
				Texture.MagnificationFilter.Bilinear, true);
		t1.setStoreTexture(true);
		ts.setTexture(t1, 0);

		Texture t2 = TextureManager.loadTexture(SimpleTerrainManager.class
				.getClassLoader().getResource("textures/Detail.jpg"),
				Texture.MinificationFilter.Trilinear,
				Texture.MagnificationFilter.Bilinear);

		ts.setTexture(t2, 1);
		t2.setWrap(Texture.WrapMode.Repeat);

		t1.setApply(Texture.ApplyMode.Combine);
		t1.setCombineFuncRGB(Texture.CombinerFunctionRGB.Modulate);
		t1.setCombineSrc0RGB(Texture.CombinerSource.CurrentTexture);
		t1.setCombineOp0RGB(Texture.CombinerOperandRGB.SourceColor);
		t1.setCombineSrc1RGB(Texture.CombinerSource.PrimaryColor);
		t1.setCombineOp1RGB(Texture.CombinerOperandRGB.SourceColor);

		t2.setApply(Texture.ApplyMode.Combine);
		t2.setCombineFuncRGB(Texture.CombinerFunctionRGB.AddSigned);
		t2.setCombineSrc0RGB(Texture.CombinerSource.CurrentTexture);
		t2.setCombineOp0RGB(Texture.CombinerOperandRGB.SourceColor);
		t2.setCombineSrc1RGB(Texture.CombinerSource.Previous);
		t2.setCombineOp1RGB(Texture.CombinerOperandRGB.SourceColor);
		cull = cs;
		tex = ts;
	}

}
