package com.l2client.gui;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.JButton;
import javax.swing.JDesktopPane;
import javax.swing.JInternalFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import com.jme.input.InputHandler;
import com.jme.renderer.Renderer;
import com.jme.scene.Node;
import com.jme.scene.Spatial;
import com.jme.system.DisplaySystem;
import com.jmex.awt.swingui.JMEDesktop;
import com.l2client.gui.dialogs.CharCreateJPanel;
import com.l2client.gui.dialogs.ChatPanel;
import com.l2client.gui.dialogs.GameServerJPanel;
import com.l2client.gui.dialogs.MinMaxListener;
import com.l2client.gui.dialogs.TransparentLoginPanel;
import com.l2client.gui.transparent.TransparentInternalFrame;
import com.l2client.model.jme.SceneRoot;
import com.l2client.model.network.GameServerInfo;


/**
 * GuiController for shuffling GUI elements around. This should just be used to hook up the
 * created gui components onto the desktop and removing the from it. No direct user actions should be defined here
 * 
 * Specific user actions should be added to the components in the @see GameController which should
 * be more specific what should be done, and also should link any needed data in.
 * 
 * used as a singleton by calling GuiController.getInstance()
 * 
 * GuiController glues the swing gui components and the JME Desktop system together, if you use
 * a different rendering system you will have to replace this on with corresponding gui calls
 * 
 * All window based gui components should use JInternalFrame to be placd on.
 */
public class GuiController {

	private static GuiController instance = null;
	private JMEDesktop jmeDesktop;
	private Node desktopNode;
	private GuiController(){	
	}
	
	public static GuiController getInstance(){
		if(instance!= null)
			return instance;
		else{
			instance = new GuiController();
			return instance;
		}
	}
	
	/**
	 * Initialize the gui controller, creates a new JMEDesktop, attaches this one to the @see SceneRoot without anything visible
	 * 
	 * @param root The SceneRoot to be used for attaching the desktop node
	 * @param input The default input handler to be used (will be put on the base stack of the InputController)
	 */
	public void initialize(SceneRoot root, InputHandler input) {
		//basic input handler
		InputController.getInstance().setInputUpdater(input);
		this.jmeDesktop = new JMEDesktop("test internalFrame");
		final DisplaySystem display = DisplaySystem.getDisplaySystem();
		// FIXME input should be set to a normal text handler without any
		// special input actions (try typing t for an example)
		jmeDesktop.setup(display.getWidth(), display.getHeight(), false, input);
		jmeDesktop.setLightCombineMode(Spatial.LightCombineMode.Off);

		desktopNode = new Node("desktop node");
		desktopNode.attachChild(jmeDesktop);
		root.attachDesktop(desktopNode);

		desktopNode.getLocalTranslation().set(display.getWidth() / 2,
				display.getHeight() / 2, 0);
		desktopNode.getLocalScale().set(1, 1, 1);
		desktopNode.setRenderQueueMode(Renderer.QUEUE_ORTHO);
		desktopNode.setCullHint(Spatial.CullHint.Never);
	}
	
	/**
	 * Used to display a username password entry dialog. @see UserPasswordJPanel
	 * A name field, a masked password entry field and an ok and cancel button
	 * The dialog will be removed automatically on login/cancel
	 * 
	 * wire actions in the <b>returned</b> control
	 * 
		// action that gets executed in the update thread:
		pan.addLoginActionListener(new JMEAction("my action", input) {
			public void performAction(InputActionEvent evt) {
				// this gets executed in jme thread
				// do 3d system calls in jme thread only!
				initNetwork(pan.getUsername(), pan.getPassword());
			}
		});
	 *
	 *
	 * @param input The input handler to handle the input on focus
	 * @return the initialized dialog for action wireing
	 */
	//TODO store username in the config and preenter it for the user on next display
	public TransparentLoginPanel displayUserPasswordJPanel(final InputHandler input){

		final JDesktopPane desktopPane = jmeDesktop.getJDesktop();
		desktopPane.removeAll();

		final JInternalFrame internalFrame = new TransparentInternalFrame();
		internalFrame.setLocation(500, 200);
		final TransparentLoginPanel pan = new TransparentLoginPanel();
		//FIXME if pan is used it is the one to be removed on close
		//how do we do dragging of window?
		pan.setTransparency(0.8f);
//		internalFrame.add(pan);
//		internalFrame.setSize(new java.awt.Dimension(200, 140));
//		internalFrame.pack();
//		internalFrame.setVisible(true);
//		desktopPane.add(internalFrame);
		
		//interresting feature too, no borders, just the panel, works too and looks ok without borders
		pan.setLocation(500, 200);
		desktopPane.add(pan);
		
		wireInputSwitch(input, pan);
		
		//these are the actions for the gui, thy do not define what should be executed on login/cancel
		 // standard swing action:
		pan.addCancelActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				// this gets executed in swing thread
				// alter swing components only in swing thread!
				internalFrame.setVisible(false);
				desktopPane.remove(internalFrame);
			}
		});
		pan.addLoginActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				// this gets executed in swing thread
				// alter swing components ony in swing thread!
				internalFrame.setVisible(false);
				desktopPane.remove(internalFrame);
			}
		});

		desktopPane.repaint();
		desktopPane.revalidate();
		return pan;
	}
	
	/**
	 * Creates a window with a gameserver list showing a status info line for each server,
	 * once the user selects a server the login button is enabled. @see GameServerJPanel
	 * The dialog will be removed automatically on cancel or select
	 * @param input			InputHandler to be used with this GUI
	 * @param serverInfos	The GameServerInfo to be displayed 
	 * @return The created gui component @see GameServerJPanel
	 */
	public GameServerJPanel displayServerSelectionJPanel(final InputHandler input, final GameServerInfo[] serverInfos){

		final JDesktopPane desktopPane = jmeDesktop.getJDesktop();
		desktopPane.removeAll();

		final JInternalFrame internalFrame = new TransparentInternalFrame();

		internalFrame.setLocation(300, 50);
		final GameServerJPanel pan = new GameServerJPanel(serverInfos);
		internalFrame.add(pan);
		
		internalFrame.setVisible(true);
		internalFrame.setSize(new java.awt.Dimension(400, 300));
		internalFrame.pack();

		desktopPane.add(internalFrame);
		
		wireInputSwitch(input, pan);
		
		 // standard swing action:
		pan.addCancelActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				// this gets executed in swing thread
				// alter swing components only in swing thread!
				internalFrame.setVisible(false);
				desktopPane.remove(internalFrame);
			}
		});
		pan.addSelectActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				// this gets executed in swing thread
				// alter swing components ony in swing thread!
				internalFrame.setVisible(false);
				desktopPane.remove(internalFrame);
			}
		});

		desktopPane.repaint();
		desktopPane.revalidate();
		return pan;
	}
	
	/**
	 * Creates a window with a gameserver list showing a status info line for each server,
	 * once the user selects a server the login button is enabled. @see GameServerJPanel
	 * The dialog will be removed automatically on cancel or select
	 * @param input			InputHandler to be used with this GUI
	 * @param serverInfos	The GameServerInfo to be displayed 
	 * @return The created gui component @see GameServerJPanel
	 */
	public ChatPanel displayChatJPanel(final InputHandler input){

		final JDesktopPane desktopPane = jmeDesktop.getJDesktop();
		desktopPane.removeAll();

		final JInternalFrame internalFrame = new TransparentInternalFrame();		
		final ChatPanel pan = new ChatPanel();
		internalFrame.add(pan);
		internalFrame.setVisible(true);
		internalFrame.setSize(new java.awt.Dimension(320, 250));
		internalFrame.pack();
		internalFrame.setLocation(0,desktopPane.getHeight()-internalFrame.getHeight());


		desktopPane.add(internalFrame);
		
		MinMaxListener mima = new MinMaxListener(internalFrame, "Chatwindow", pan);
		pan.addMouseListener(mima);
		
		wireInputSwitch(input, pan);

		desktopPane.repaint();
		desktopPane.revalidate();
		return pan;
	}
	
	
	/**
	 * Draft of a character creation panel, a race field to choose from, a name, a class, a create and cancel button
	 * The dialog will be removed automatically on cancel, but stay open on create
	 * @param input The InputHandler to be used with this gui
	 * @return		The created @see CharCreateJPanel where the game logic can be added to
	 */
	public CharCreateJPanel displayCharCreateJPanel(final InputHandler input){

		final JDesktopPane desktopPane = jmeDesktop.getJDesktop();
		desktopPane.removeAll();
		
		final JInternalFrame internalFrame = new JInternalFrame();

		internalFrame.setLocation(20, 20);
		internalFrame.setResizable(false);
		internalFrame.setFrameIcon(null);
		final CharCreateJPanel pan = new CharCreateJPanel();
		internalFrame.add(pan);
		
		internalFrame.setVisible(true);
		internalFrame.setSize(new java.awt.Dimension(200, 180));
		internalFrame.pack();

		desktopPane.add(internalFrame);
		
		wireInputSwitch(input, pan);
		
		 // standard swing action:
		pan.addCancelActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				// this gets executed in swing thread
				// alter swing components only in swing thread!
				internalFrame.setVisible(false);
				desktopPane.remove(internalFrame);
			}
		});
		
		//nothing done in swing thread, pane stays open and will be closed on
		//create ok or cancel case only!
//		pan.addCreateActionListener(new ActionListener() {
//			public void actionPerformed(ActionEvent e) {
//				// this gets executed in swing thread
//				// alter swing components ony in swing thread!
//				internalFrame.setVisible(false);
//				desktopPane.remove(internalFrame);
//			}
//		});

		desktopPane.repaint();
		desktopPane.revalidate();
		return pan;
	}

	/**
	 * Convenience functionality to be able to display a modal Error Dialog based on a JOptionPane.
	 * The dialog will be removed automatically on close,calls SwingUtilities.invokeLater by itself
	 * 
	 * @param messageText the error message to be displayed
	 */
	public void showErrorDialog(final String messageText) {
		SwingUtilities.invokeLater(new Runnable() {

			public void run() {
				final JDesktopPane desktopPane = jmeDesktop.getJDesktop();
				final JInternalFrame modalDialog = new JInternalFrame("Error");

				JOptionPane optionPane = new JOptionPane(messageText, JOptionPane.ERROR_MESSAGE);
				modalDialog.getContentPane().add(optionPane);
				jmeDesktop.setModalComponent(modalDialog);
				desktopPane.add(modalDialog, 0);
				modalDialog.setVisible(true);
				modalDialog.setSize(modalDialog.getPreferredSize());
				modalDialog
						.setLocation((desktopPane.getWidth() - modalDialog
								.getWidth()) / 2,
								(desktopPane.getHeight() - modalDialog
										.getHeight()) / 2);
				jmeDesktop.setFocusOwner(optionPane);

				optionPane.addPropertyChangeListener(
						JOptionPane.VALUE_PROPERTY,
						new PropertyChangeListener() {
							public void propertyChange(PropertyChangeEvent evt) {
								modalDialog.setVisible(false);
								jmeDesktop.setModalComponent(null);
								desktopPane.remove(modalDialog);
							}
						});
			}
		});
    }

	/**
	 * removes all gui components (but not buttons)
	 */
	//FIXME does not remove buttons though
	public void removeAll() {
		SwingUtilities.invokeLater(new Runnable() {

			public void run() {
				final JDesktopPane desktopPane = jmeDesktop.getJDesktop();
				desktopPane.removeAll();
			}
		});
	}
	
	/**
	 * Convenience functionality to be able to display a modal info Dialog based on a JOptionPane.
	 * The dialog will be removed automatically on close, calls SwingUtilities.invokeLater by itself
	 * @param messageText the info message to be displayed
	 */
	//TODO could be refactored to be one code with showErrorDialog
	public void showInfoDialog(final String messageText) {
		SwingUtilities.invokeLater(new Runnable() {

			public void run() {
				final JDesktopPane desktopPane = jmeDesktop.getJDesktop();
				final JInternalFrame modalDialog = new JInternalFrame("Info");

				JOptionPane optionPane = new JOptionPane(messageText, JOptionPane.INFORMATION_MESSAGE);
				modalDialog.getContentPane().add(optionPane);
				jmeDesktop.setModalComponent(modalDialog);
				desktopPane.add(modalDialog, 0);
				modalDialog.setVisible(true);
				modalDialog.setSize(modalDialog.getPreferredSize());
				modalDialog
						.setLocation((desktopPane.getWidth() - modalDialog
								.getWidth()) / 2,
								(desktopPane.getHeight() - modalDialog
										.getHeight()) / 2);
				jmeDesktop.setFocusOwner(optionPane);

				optionPane.addPropertyChangeListener(
						JOptionPane.VALUE_PROPERTY,
						new PropertyChangeListener() {
							public void propertyChange(PropertyChangeEvent evt) {
								modalDialog.setVisible(false);
								jmeDesktop.setModalComponent(null);
								desktopPane.remove(modalDialog);
							}
						});
			}
		});
    }

	/**
	 * creates a button on the desktop, caller is responsible to clean up the buttons (removeButton or removeAll)
	 * @param name
	 * @return
	 */
	public JButton displayButton(String name) {
		final JButton b = new JButton(name);
		final JDesktopPane desktopPane = jmeDesktop.getJDesktop();
		desktopPane.add(b);
		b.setVisible(true);
		desktopPane.repaint();
		desktopPane.revalidate();
		return b;
	}
	
	/**
	 * removes the passed buttons from the desktop, calls SwingUtilities.invokeLater by itself
	 * @param b
	 */
	public void removeButton(final JButton[] b) {
		SwingUtilities.invokeLater(new Runnable() {

			public void run() {
				final JDesktopPane desktopPane = jmeDesktop.getJDesktop();
				for (JButton j : b)
					desktopPane.remove(j);

				desktopPane.repaint();
				desktopPane.revalidate();
			}
		});
	}
	
	private void wireInputSwitch(final InputHandler input, Component comp){
		comp.addMouseListener(new MouseListener() {
			
			@Override
			public void mouseReleased(MouseEvent e) {
			}
			
			@Override
			public void mousePressed(MouseEvent e) {
			}
			
			@Override
			public void mouseExited(MouseEvent e) {
				InputController.getInstance().popInput();
			}
			
			@Override
			public void mouseEntered(MouseEvent e) {
				InputController.getInstance().pushInput(input);
			}
			
			@Override
			public void mouseClicked(MouseEvent e) {				
			}
		});
	}
}
