package com.l2client.model.jme;

import com.jme.scene.Node;
import com.jme.scene.state.LightState;

/**
 * The scene root is basically a compound of separate scene nodes used for rendering the visual components
 * The hierarchy for rendering is normally processed along the tree. The SceneRoot is attached to the top
 * node in the rendering hierarchy and provides getters and setters for specific client nodes
 * 
 * top root
 *  + desktopNode
 *  + scene
 *  + charRoot
 *  
 *  The desktopNode is the root for all gui components, so with one call to remove all you can clean away all gui, etc.
 *  The scene is the top node for all to be rendered 3d models (currently no difference between dynamic and static geoms)
 *  The charRoot is the top node for all character related visual rendering
 *
 */
public class SceneRoot extends Node {

	private static final long serialVersionUID = 6925161101907738054L;
	/**
	 * root node for all desktop nodes, will be set by GuiController, do not detach from parent, otherwise do wah you want with it.
	 */
	private Node desktopNode = null;
	/**
	 * root node for all scene nodes, do not detach from parent, otherwise do what you want with it.
	 */
	private Node scene = new Node("scene root");
	/**
	 * root node for character related nodes, do not detach from parent, otherwise do what you want with it.
	 */
	private Node charRoot = new Node("player root");
	/**
	 * the light state to be used for the scene
	 */
	private LightState lightState = null;
	
	public void init(LightState state){
		lightState = state;
		attachChild(scene);
		attachChild(charRoot);
		scene.setCullHint(CullHint.Dynamic);
	}
	
	/**
	 * Removes all nodes below charRoot
	 */
	//TODO not used at the moment, should be called at least on exit to the lobby
	public void cleanupChar(){
		charRoot.detachAllChildren();
		charRoot.updateModelBound();
	}
	
	/**
	 * removes all nodes below scene
	 */
	public void cleanupScene(){
		scene.detachAllChildren();
		scene.updateModelBound();
	}
	
	/**
	 * Attaches the desktop node to the scene root @see GuiController.initialize
	 * @param deskNode the root node to be used for gui elements
	 */
	public void attachDesktop(Node deskNode){
		this.desktopNode = deskNode;
		attachChild(desktopNode);
	}

	/**
	 * Returns the scene root for inidividual manipulation. 
	 * do not detach from parent, otherwise do what you want with it.
	 *
	 * @return a Node representing the scene root
	 */
	public final Node getScene() {
		return scene;
	}

	/**
	 * Returns the light state of the scene for individual manipulation
	 * do not detach from parent, otherwise do what you want with it.
	 * 
	 * @return the LightState of he scene root
	 */
	public final LightState getLightState() {
		return lightState;
	}

	/**
	 * Returns the character root light state of the scene for individual manipulation
	 * do not detach from parent, otherwise do what you want with it.
	 * 
	 * @return the node representing the character root
	 */
	public final Node getCharRoot() {
		return charRoot;
	}
}
