package com.l2client.model.jme;

import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.jme.bounding.BoundingBox;
import com.jme.math.FastMath;
import com.jme.math.Vector3f;
import com.jme.scene.Controller;
import com.jme.scene.Node;
import com.jme.scene.Spatial;
import com.jme.util.resource.RelativeResourceLocator;
import com.jme.util.resource.ResourceLocatorTool;
import com.jmex.model.ModelFormatException;
import com.jmex.model.ogrexml.MaterialLoader;
import com.jmex.model.ogrexml.MeshCloner;
import com.jmex.model.ogrexml.OgreLoader;
import com.jmex.model.ogrexml.anim.MeshAnimationController;
import com.l2client.model.network.NewCharSummary;
import com.l2client.util.TextLabel2D;

/**
 * visual representation of a model in 3d space
 * has a controller for 
 * 0) animation
 * 1) movement (optionally) 
 * 
 * should be subclassed for individual representations of npc's characters, etc.
 * A NpcBuilder should construct the individual visual specific detail representations.
 * 
 * Currently all models are based on the same visual model, a modelcache or AssetManager should be used to speed loading of new models (which should be done by a builder/factory patterns anyway)
 *
 */
//TODO animation controller, rigging of actions against animations
public class VisibleModel extends Node {

	private static final long serialVersionUID = 1L;

	Logger logger = Logger.getLogger(VisibleModel.class.getName());

	NewCharSummary charSelection;

	/**
	 * In the demo all models look alike, so we store a base node
	 */
	private static Node baseNode = null;
	/**
	 * The final individual node, including the name of then object above its head
	 */
	private Node vis = null;

	/**
	 * Internal class representing an simple movement controller for moving models around.
	 * The controller is attached on the visual and moves the visual around, on reaching its goal
	 * the controller removes itself, no turning at the moment
	 *
	 */
	private class MoveController extends Controller{

		private static final long serialVersionUID = 1L;
		/// target position 
		private Vector3f target = null;
		/// direction to travel along
		private Vector3f direction = null;
		/// node to be moved
		private Spatial parent = null;
		
		MoveController(Spatial _parent, Vector3f targetPos, float speed) {
			target = targetPos;
			parent = _parent;
			setSpeed(speed);
			if(parent != null && target != null)
				direction = target.subtract(parent.getLocalTranslation()).normalizeLocal();
		}
		
		/**
		 * Called during rendering, updates parent local Translation and removes itself on reaching the goal.
		 * The target is moved time * speed * normalized direction vector.
		 */
		@Override
		public void update(float time) {
			if(target != null){
				Vector3f parentPos = parent.getLocalTranslation();
	            //target reached then remove self
	            if (target.subtract(parentPos).lengthSquared() <= (0.1f)) {
	                parent.setLocalTranslation(target);
	                this.setActive(false);
	            	parent.removeController(this);	            	
	                this.target = null;
	                this.direction = null;
	                this.parent = null;
	                return;
	            }
	            //TODO the npcdata should be updated too
	            //move
	            //TODO turn towards target direction and move
	            parent.setLocalTranslation(parentPos.add(direction.mult(time * getSpeed())));
			}
		}
	};

	/**
	 * Constructor for a vismodel
	 * 
	 * @param sel the NewCharSummary this model should be based on (used for assembling the final visual representation)
	 */
	public VisibleModel(NewCharSummary sel) {
		charSelection = sel;
	}
	
	
	/**
	 * In the demo we drop out the z value (height) as all walk on the plane for simplicity
	 */
	//FIXME actually ignore height values
	@Override
	public void setLocalTranslation(float x, float y, float z){
		super.setLocalTranslation(x, y, 0f);
	}
	
	/**
	 * In the demo we drop out the z value (height) as all walk on the plane for simplicity
	 */
	//FIXME actually ignore height values
	@Override
	public void setLocalTranslation(Vector3f vec){
		super.setLocalTranslation(vec.x, vec.y, 0f);
	}

	/**
	 * Creates the visual if needed by loading it and places the name label above its head
	 */
	public void attachVisuals() {
		if (vis != null)
			attachChild(vis);
		else {
			createVisuals();
			if (vis != null)
				attachChild(vis);
		}
		if(vis != null){
			//done here for triggering an update
			updateGeometricState(0, true);
			// needed here otherwise will have dummy renderstates
			updateRenderState();
		}
		if (vis != null && charSelection.name != null) {
			//			vis.updateModelBound();
			//FIXME is this label ok?
			TextLabel2D t = new TextLabel2D(charSelection.name);
			Node n = t.getBillboard(1.0f);
			//FIXME this can blow wild, if we dont check for the real class
			n.setLocalTranslation(0f, 0f, (((BoundingBox) vis.getWorldBound())
					.getExtent(null).z * 2.0f) + 2f);
			vis.attachChild(n);
		}
	}
	
	/**
	 * Builds the visual representation of the model by loading it currently.
	 * The anim controller seems to be missing from the jme stored version, so currently 
	 * we load the model from the plain ogre xml definition (which is a bit slower)
	 * 
	 * 
	 * @return the loaded model on a @see Node
	 */
	//FIXME move to a builder, whcih should know what assets to load for which visual (stored in the DB)
	protected Node createVisuals() {

		if (charSelection != null) {
			if (baseNode == null) {
				
//				//does not restore the animations !?!				
//				try {
//					baseNode = (Node) BinaryImporter.getInstance().load(VisibleModel.class.getClassLoader().getResource(
//							"troll2/troll.jme"));
////					BinaryExporter.getInstance().save(baseNode, new File("/troll2/troll.jme"));
//				} catch (IOException e) {
//					// TODO Auto-generated catch block
//					e.printStackTrace();
//				}

				OgreLoader loader = new OgreLoader();
				MaterialLoader matLoader = new MaterialLoader();
				String matUrlString = "/troll2/troll.xml.material";
				String ninjaMeshUrlString = "/troll2/troll.xml.mesh.xml";

				try {
					URL matURL = ResourceLocatorTool.locateResource(
							ResourceLocatorTool.TYPE_TEXTURE, matUrlString);
					URL meshURL = ResourceLocatorTool.locateResource(
							ResourceLocatorTool.TYPE_MODEL, ninjaMeshUrlString);

					if (meshURL == null)
						throw new IllegalStateException(
								"Required runtime resource missing: "
										+ ninjaMeshUrlString);
					if (matURL == null)
						throw new IllegalStateException(
								"Required runtime resource missing: "
										+ matUrlString);
					try {
						ResourceLocatorTool.addResourceLocator(
								ResourceLocatorTool.TYPE_TEXTURE,
								new RelativeResourceLocator(matURL));
						// This causes relative references in the .material file to
						// resolve to the same dir as the material file.
						// Don't have to set up a relative locator for TYPE_MODEL
						// here, because OgreLoader.loadModel() takes care of that.
					} catch (URISyntaxException use) {
						// Since we're generating the URI from a URL we know to be
						// good, we won't get here.  This is just to satisfy the
						// compiler.
						throw new RuntimeException(use);
					}
					matLoader.load(matURL.openStream());
					if (matLoader.getMaterials().size() > 0)
						loader.setMaterials(matLoader.getMaterials());

					baseNode = (Node) loader.loadModel(meshURL);
				} catch (Exception ex) {
					logger.log(Level.SEVERE, null, ex);
				}
			}
			if (baseNode != null) {
				//FIXME modelconverter should already have set this one, this is not the case -> NPE
				baseNode.setModelBound(new BoundingBox());
				baseNode.updateModelBound();
				vis = MeshCloner.cloneMesh(baseNode);

				if (vis.getControllerCount() < 1)
					throw new IllegalStateException(
							"Vis animations are missing");
				MeshAnimationController animControl = (MeshAnimationController) vis
						.getController(0);
				animControl.setAnimation("idle");
				animControl.setCurTime(animControl.getAnimationLength("idle")
						* FastMath.nextRandomFloat());
			}
		}

		return vis;
	}
	
	/**
	 * Adds a movement controller to move the model to the specified x,y coordinates (z is 0 here)
	 * A previous move controller is removed
	 * @param toX target position in world coords x
	 * @param toY target position in world coords y
	 * @param toZ ignored
	 * @param speed   speed per second to be used for traveling
	 */
	// FIXME ignores height at the moment!
	public void initMoveTo(float toX, float toY, float toZ, float speed) {
		if (speed > 0f) {
			Controller rem = null;
			for (Controller con : getControllers()) {
				if (con instanceof MoveController) {
					con.setActive(false);
					rem = con;
				}
			}
			if (rem != null)
				removeController(rem);
			addController(new MoveController(this, new Vector3f(toX, toY, 0), speed));
		}
	}
}
