package com.l2client.util;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.ConvolveOp;
import java.awt.image.Kernel;
import java.util.Arrays;

import com.jme.image.Texture;
import com.jme.renderer.Renderer;
import com.jme.scene.BillboardNode;
import com.jme.scene.Spatial.LightCombineMode;
import com.jme.scene.shape.Quad;
import com.jme.scene.state.BlendState;
import com.jme.scene.state.TextureState;
import com.jme.system.DisplaySystem;
import com.jme.util.TextureManager;

public class TextLabel2D {

	private String text;

	private float blurIntensity = 0.1f;

	private int kernelSize = 5;

	private ConvolveOp blur;

	private Color foreground = new Color(1f, 1f, 1f);

	private Color background = new Color(0f, 0f, 0f);

	private float fontResolution = 40f;

	private int shadowOffsetX = 2;

	private int shadowOffsetY = 2;

	private Font font;

	public TextLabel2D(String text) {
		this.text = text;
		updateKernel();
		setFont(Font.decode("Sans PLAIN 40"));
	}

	public void setFont(Font font) {
		this.font = font;
	}

	public void setShadowOffsetX(int offsetPixelX) {
		shadowOffsetX = offsetPixelX;
	}

	public void setShadowOffsetY(int offsetPixelY) {
		shadowOffsetY = offsetPixelY;
	}

	public void setBlurSize(int kernelSize) {
		this.kernelSize = kernelSize;
		updateKernel();
	}

	public void setBlurStrength(float strength) {
		this.blurIntensity = strength;
		updateKernel();
	}

	public void setFontResolution(float fontResolution) {
		this.fontResolution = fontResolution;
	}

	private void updateKernel() {
		float[] kernel = new float[kernelSize * kernelSize];
		Arrays.fill(kernel, blurIntensity);
		blur = new ConvolveOp(new Kernel(kernelSize, kernelSize, kernel));
	}

	public BufferedImage getImage() {
		BufferedImage tmp0 = new BufferedImage(10, 10,
				BufferedImage.TYPE_INT_ARGB);
		Graphics2D g2d = (Graphics2D) tmp0.getGraphics();
		Font drawFont = font.deriveFont(fontResolution);
		g2d.setFont(drawFont);
		Rectangle2D b = g2d.getFontMetrics().getStringBounds(text, g2d);
		int desiredX = (int) b.getWidth() + kernelSize + 1 + shadowOffsetX;
		int desiredY = (int) b.getHeight() + kernelSize + 1 + shadowOffsetY;
		tmp0 = new BufferedImage(desiredX, desiredY,
				BufferedImage.TYPE_INT_ARGB);
		g2d = (Graphics2D) tmp0.getGraphics();
		g2d.setFont(drawFont);
		g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);
		g2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
				RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
		int textX = kernelSize / 2;
		int textY = g2d.getFontMetrics().getMaxAscent() - kernelSize / 2;
		g2d.setColor(background);
		g2d.drawString(text, textX + shadowOffsetX, textY + shadowOffsetY);
		BufferedImage ret = blur.filter(tmp0, null);
		g2d = (Graphics2D) ret.getGraphics();
		g2d.setFont(drawFont);
		g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);
		g2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
				RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
		g2d.setColor(foreground);
		g2d.drawString(text, textX, textY);
		return ret;
	}

	public Quad getQuad(float height) {
		BufferedImage img = getImage();
		int w = img.getWidth();
		int h = img.getHeight();
		float factor = height / img.getHeight();
		Quad ret = new Quad("textLabel2d", img.getWidth() * factor, img
				.getHeight()
				* factor);
		TextureState ts = DisplaySystem.getDisplaySystem().getRenderer()
				.createTextureState();
		Texture tex = TextureManager.loadTexture(img, Texture.MinificationFilter.BilinearNoMipMaps,
				Texture.MagnificationFilter.Bilinear, true);
		ts.setTexture(tex);
		ts.setEnabled(true);
		ret.setRenderState(ts);
		ret.setRenderQueueMode(Renderer.QUEUE_TRANSPARENT);
		BlendState as1 = DisplaySystem.getDisplaySystem().getRenderer().createBlendState();
        as1.setBlendEnabled( true );
        as1.setSourceFunction( BlendState.SourceFunction.SourceAlpha );
        as1.setDestinationFunction( BlendState.DestinationFunction.OneMinusSourceAlpha );
		ret.setRenderState(as1);
		ret.setLightCombineMode(LightCombineMode.Off);
		ret.updateRenderState();
		return ret;
	}

	public BillboardNode getBillboard(float height) {
		BillboardNode bb = new BillboardNode("bb");
		Quad q = getQuad(height);
		bb.attachChild(q);
		return bb;
	}

	public void setForeground(Color foreground) {
		this.foreground = foreground;
	}

	public void setBackground(Color background) {
		this.background = background;
	}
}