#ifndef I3D_POOL_H
#define I3D_POOL_H
//------------------------------------------------------------------------------
/**
	@class Pool
	@ingroup 3DInfernoBase

	@brief a pool of objects in memory
*/
#include "kernel/ntypes.h"

//------------------------------------------------------------------------------
template<class PTYPE> class Pool
{
public:
	Pool(const int Poolsize,const bool Grow = true);
	Pool();
	~Pool();

	/// get next free item from pool
	PTYPE * GetNext(void);
	/// release item to pool
	void Release(PTYPE * ItemToBeDeleted);
	/// clear pool without calling destructors
	void Clear();
	   
private:
	typedef PTP (PTYPE*);
	/// Allocate next pool and  attach it as a child to this one
	Pool(Pool * const PrevPool);
	void Init()
	{
		mPool = n_new (PTYPE [mPoolSize]);
		//mFreeStack = n_new sizeof(PTYPE*) [mPoolSize];
		mFreeStack = n_new (PTYPE* [mPoolSize]);
		mAvailable = mPool;
		mLast = &mPool[mPoolSize-1];// +(sizeof(PTYPE) * mPoolSize);
	};
	/// Parent pool, is NULL in Interface starter NOT REALY USED
	Pool *	mPrev;
	/// Next child pool in list
	Pool *	mNext;
	/// # of total items in pool
	int		mPoolSize;
	/// Next free item in pool
	PTYPE *	mAvailable;
	/// Last in Pool
	PTYPE *	mLast;
	/// Top Item on free stack
	int		mTop;
	/// Is pool allowed to grow (and attach child pools to itself)
	bool	mGrowable;
	/// Pool of items
	PTYPE *	mPool;
	/// Stack of deleted items
	PTYPE **	mFreeStack;
};


template <class PTYPE> 
Pool<PTYPE>::Pool():mPrev(NULL),mNext(NULL),mPoolSize(32),mTop(NULL),mGrowable(true)
{
	Init();		
}

template <class PTYPE> 
Pool<PTYPE>::Pool(const int Poolsize, const bool Grow): 
		mPrev(NULL),
		mNext(NULL),
		mPoolSize(Poolsize),
		mTop(NULL),
		mGrowable(Grow)
{
	Init();				
}

template <class PTYPE> 
Pool<PTYPE>::Pool(Pool * const Prev):
		mPrev(Prev),
		mNext(NULL),
		mPoolSize(Prev->mPoolSize),
		mTop(NULL),
		mGrowable(true)// or else we could not have been called
{
	Init();			
}

template <class PTYPE> 
Pool<PTYPE>::~Pool()
{
	if(mNext)
		mNext->~Pool();
	if(mPool)
		n_delete_array(mPool);
	if(mFreeStack)
		n_delete_array(mFreeStack);
	mTop = 0;
	mAvailable = NULL;
};

template <class PTYPE> 
PTYPE * Pool<PTYPE>::GetNext()
{
	if(mTop  != 0)
		return mFreeStack[--mTop];
	else
	{
		if(mAvailable < mLast)
		{
			PTYPE * ret = mAvailable;
			mAvailable++;
			return ret;
		}
		else
		{
			if(mNext)
			{
				return mNext->GetNext();
			}
			else
			{
				if(mGrowable)
				{
					mNext = n_new( Pool(this));
					if (mNext)
					{
						return mNext->GetNext();
					}
					else
					{
						n_error("Pool run out of new mem !");
						return 0;
					}
				}
				else
				{
					n_error("Pool out of memory !"); // or barf 
					return NULL;
				}
			}
		}
	}
}

template <class PTYPE> 
void Pool<PTYPE>::Clear()
{
	if(mPool)
	{
		mTop = 0;
		mAvailable = mPool;
		if(mNext)
			mNext->Clear();
	}
};

template <class PTYPE> 
void Pool<PTYPE>::Release(PTYPE * FreeItem)
{
	if(FreeItem)
	{
		//is it is NOT in our interval
		if((FreeItem < mPool )||(FreeItem >= mLast))
		{
			if(mNext)
				mNext->Release(FreeItem);
			else
			{	
				n_error("Release of memory from foreign data segment!"); // or barf 
			}
		}
		else
		{
			mFreeStack[mTop++] = FreeItem;
			if(mPrev && (! mNext )&& (mTop >= (mPoolSize-1)))
			{
				mPrev->mNext = NULL;
				n_delete(this);
			}
		}
	}
};

//------------------------------------------------------------------------------
#endif
