#ifndef N_MD2LOADER_H
#define N_MD2LOADER_H
//------------------------------------------------------------------------------
/** 
    @class nMD2Loader
	@ingroup 3DInfernoMD2ipol
	
    @brief Loads a md2 (quake 2 file format) mesh file into user provided vertex and index buffers.
    
    (C) 2004 tom@3D-Inferno.com
*/
#include "gfx2/nmeshloader.h"
#include "gfx2/nmesh2.h"
#include "kernel/nfile.h"
#include "anim/3di_nblendanimloader.h"
#include "anim/3di_md2.h"

struct nMD2Animation
{
	char name[32]; // 16 * 2 for safety
	unsigned int frames;
	unsigned int firstframe;
};




//------------------------------------------------------------------------------
class nMD2Loader : public nBlendAnimLoader
{
public:
    /// constructor
    nMD2Loader();
    /// destructor
    ~nMD2Loader();
    /// open file and read header data
    virtual bool Open(nFileServer2* fs);
    /// close the file
    virtual void Close();
    /// read vertex data
    virtual bool ReadVertices(void* buffer, unsigned int bufferSize);
    /// read index data
    virtual bool ReadIndices(void* buffer, unsigned int bufferSize);
    /// read edge data
    virtual bool ReadEdges(void* buffer, unsigned int bufferSize);
	/// load the animation definitions this anims will be filled and then belongs to the caller of this function. so clean it up !
	virtual bool ReadAnimations(nBlendAnimations*anims, unsigned int animoCount);
	/// Return number of animations in file
	virtual int GetAnimationCount();
	//virtual int GetVertexWidth() const{};
	//virtual int GetVertexComponents() const {};

	/// the different animations in the md2 file (run, stand, attack1 etc.)
	nMD2Animation			*Anims;
	/// number of different animations
	unsigned int			m_anims;
	/// summ of animation frames for all animations
	unsigned int			m_totalframes;
private:
	/// temp index list
	make_index_list			*m_index_list;
	/// temp frame list
	make_frame_list			*m_frame_list;
	/// file header
	SMD2Header				modelheader;
	/// skins for model
	char					g_skins[MAX_MD2SKINS][64];


};

//------------------------------------------------------------------------------
/**
*/
inline
nMD2Loader::nMD2Loader()
{
	m_frame_list = NULL;
	m_index_list = NULL;
	m_anims = numTriangles = numVertices = m_totalframes = 0;
	Anims = NULL;
}

//------------------------------------------------------------------------------
/**
*/
inline
nMD2Loader::~nMD2Loader()
{
	// clean up allocated stuff
	if(m_frame_list != NULL)
		for(int i = 0; i < modelheader.num_frames; i++)
			n_delete_array(m_frame_list[i].vertex);

	if(m_index_list)
		n_delete_array(m_index_list);
	if(m_frame_list)
		n_delete_array(m_frame_list);
	if(Anims)
		n_delete_array(Anims);
	
	m_frame_list = NULL;
	m_index_list = NULL;
}

//------------------------------------------------------------------------------
/**
*/
inline
bool
nMD2Loader::Open(nFileServer2* fs)
{
    n_assert(!this->file);
    n_assert(fs);

    this->fileServer = fs;
    this->file = this->fileServer->NewFileObject();
    n_assert(this->file);

    // open the file
    if (!this->file->Open(this->filename.Get(), "r"))
    {
        n_printf("nMD2Loader: could not open file '%s'!\n", this->filename.Get());
        this->Close();
        return false;
    }

    char		g_skins[MAX_MD2SKINS][64];
    dstvert_t	base_st[MAX_VERTS];
    byte		buffer[MAX_VERTS*4+128]; // don't ask me why, legacy code

    dtriangle_t     tri;
    daliasframe_t	*out;
	
    // read header
	this->file->Read(&modelheader, sizeof(modelheader));
	if(modelheader.ident != 844121161) // IDP2 not a quake 2 file
	{
		n_printf("File <%s> is not in MD2 format.",this->filename.Get());
		this->file->Close();
		return false;
	}

	
    modelheader.framesize = (int)&((daliasframe_t *)0)->verts[modelheader.num_xyz];

	this->vertexComponents = nMesh2::Coord | nMesh2::Normal | nMesh2::Uv0;
	//dumb specify vertex comp by nMesh but use hardcoded value for Width should create
	//enum for widths in nMesh2!
	this->vertexWidth = 8;
	// these values are for only a single frame !!!
	this->numVertices = modelheader.num_tris * 3 * modelheader.num_frames;
	this->numTriangles = modelheader.num_tris * modelheader.num_frames;		
	this->numIndices = modelheader.num_tris * 3;
	//md2 file does not provide edges so far and we will not be computing this at the moment
	this->numEdges = 0;

	//make_index_list *m_index_list = n_new make_index_list [modelheader.num_tris];
	m_index_list = n_new( make_index_list [modelheader.num_tris]);
	//make_frame_list *m_frame_list = n_new make_frame_list [modelheader.num_frames];
	m_frame_list = n_new( make_frame_list [modelheader.num_frames]);
	
	for( int i = 0; i < modelheader.num_frames; i++)
		m_frame_list[i].vertex = n_new( make_vertex_list [modelheader.num_xyz]);
	
	// read skin information
	this->file->Read(g_skins, modelheader.num_skins * MAX_SKINNAME);
	// read index to polymeshes
	this->file->Read(base_st, modelheader.num_st * sizeof(base_st[0]));
	
	int	max_tex_u = 0, max_tex_v = 0;

	for( i = 0; i < modelheader.num_tris; i++ ) 
	{
		// read vertices
		this->file->Read(&tri, sizeof(dtriangle_t));
		
		(m_index_list)[i].a = tri.index_xyz[2];
		(m_index_list)[i].b = tri.index_xyz[1];
		(m_index_list)[i].c = tri.index_xyz[0];
	
		// texture coordinates
		(m_index_list)[i].a_s = base_st[tri.index_st[2]].s;
		(m_index_list)[i].a_t = base_st[tri.index_st[2]].t;
		(m_index_list)[i].b_s = base_st[tri.index_st[1]].s;
		(m_index_list)[i].b_t = base_st[tri.index_st[1]].t;
		(m_index_list)[i].c_s = base_st[tri.index_st[0]].s;
		(m_index_list)[i].c_t = base_st[tri.index_st[0]].t;
		max_tex_u = n_max( max_tex_u, base_st[tri.index_st[0]].s );
		max_tex_u = n_max( max_tex_u, base_st[tri.index_st[1]].s );
		max_tex_u = n_max( max_tex_u, base_st[tri.index_st[2]].s );
		max_tex_v = n_max( max_tex_v, base_st[tri.index_st[0]].t );
		max_tex_v = n_max( max_tex_v, base_st[tri.index_st[1]].t );
		max_tex_v = n_max( max_tex_v, base_st[tri.index_st[2]].t );
	}

	// MD2 stores texture coords dependant on pixelbasis (depending on the original size of the texture)
	// we need to recalc this
	for ( i = 0; i < modelheader.num_tris; i++ ) 
	{
		m_index_list[ i ].a_s /= max_tex_u;
		m_index_list[ i ].b_s /= max_tex_u;
		m_index_list[ i ].c_s /= max_tex_u;
		m_index_list[ i ].a_t /= max_tex_v;
		m_index_list[ i ].b_t /= max_tex_v;
		m_index_list[ i ].c_t /= max_tex_v;
	}

	nMD2Animation *stlAnims = n_new( nMD2Animation [modelheader.num_frames]);
	char curr[16] = "";
	char * pnew = NULL;
	int cnt = 0;
	this->m_anims = 0;

	// read vertex data of all animations 
	for( i = 0; i < modelheader.num_frames; i++ ) 
	{
		out = (daliasframe_t *)buffer;
		this->file->Read(out, modelheader.framesize);

		if (out->name)
		{
			n_printf(" %s ",out->name);
			pnew = strpbrk(out->name, "0123456789");
			*pnew = '\0';
			if(strcmp(out->name,curr) == 0)
			{
				cnt++;
			}
			else
			{
				if(cnt != 0)
				{
					//write stuff to anim set counter to 1 and copy anim name
					this->m_anims ++;
					nMD2Animation anim;
					strcpy(anim.name, curr);
					strcpy(curr, out->name);
					anim.frames = cnt;
					anim.firstframe = i -cnt;
					stlAnims[this->m_anims -1] = anim;
					cnt = 1;
				}
				else
				{
					// only in init case (first call ) sucks should do a while and first read ahead
					strcpy(curr, out->name);
					cnt++;
				}
			}
		}
			
		for( int j = 0; j < modelheader.num_xyz; j++ ) 
		{
			(m_frame_list)[i].vertex[j].x = out->verts[j].v[0] * out->scale[0] + out->translate[0];
			(m_frame_list)[i].vertex[j].y = out->verts[j].v[1] * out->scale[1] + out->translate[1];
			(m_frame_list)[i].vertex[j].z = out->verts[j].v[2] * out->scale[2] + out->translate[2];
		}
	}
	
	nMD2Animation anim;
	this->m_anims ++;
	strcpy(anim.name, curr);
	anim.frames = cnt;
	stlAnims[this->m_anims -1] = anim;

	//init animations array
	if(this->Anims)
		n_delete_array( this->Anims);
	this->Anims = n_new( nMD2Animation[this->m_anims]);
	n_assert(this->Anims);

	for(unsigned int f = 0; f< m_anims; f++)
	{
		this->Anims[f] = stlAnims[f];
	}	

	// each frame his own meshgroup
	// this won't be used in BlendAnimations and will cause an error anyway
	// if you use it with meshbuilder as it does not obey group boundries!
	for(i = 0; i < modelheader.num_frames;i++)
	{
		nMeshGroup meshGroup;
		meshGroup.SetFirstVertex(i * modelheader.num_tris * 3);
		meshGroup.SetNumVertices(modelheader.num_tris * 3);
		meshGroup.SetFirstIndex(i*modelheader.num_tris * 3);
		meshGroup.SetNumIndices(modelheader.num_tris * 3);//FIXME we have indices butttwe don't show them as we only use one
		meshGroup.SetFirstEdge(0);
		meshGroup.SetNumEdges(0);
		this->groupArray.Append(meshGroup);
	}
	this->numGroups = modelheader.num_frames;

	this->vertexComponents = nMesh2::Coord|nMesh2::Normal|nMesh2::Texture;
	this->vertexWidth = 8;

    return true;
}

//------------------------------------------------------------------------------
/**
*/
inline
void
nMD2Loader::Close()
{
    if (this->file)
    {
        if (this->file->IsOpen())
        {
            this->file->Close();
        }
        this->file->Release();
        this->file = 0;
    }
}

//------------------------------------------------------------------------------
/**
*/
inline
bool
nMD2Loader::ReadVertices(void* buffer, unsigned int bufferSize)
{
    n_assert(buffer);
	n_assert(Anims);
	n_assert(m_frame_list);
	n_assert(m_index_list);
	n_assert(bufferSize >= (unsigned int)(modelheader.num_tris * 3 *vertexWidth))
    n_assert((this->numVertices * this->vertexWidth * sizeof(float)) == bufferSize);
    void* endOfBuffer = ((float*)buffer) + (this->numVertices * this->vertexWidth);

	float * vPtr = ((float*)buffer);
	vector3 a,b,c;
	for (int i = 0; i < modelheader.num_frames; i++ )
	{
		for( int j = 0; j < modelheader.num_tris; j++) 
		{
			//  beware y and z are flipped in MD2 format as they use different
			//  coordinate system
			a.x = m_frame_list[i].vertex[m_index_list[j].a].x;
			a.y = m_frame_list[i].vertex[m_index_list[j].a].z;
			a.z = m_frame_list[i].vertex[m_index_list[j].a].y;

			b.x = m_frame_list[i].vertex[m_index_list[j].b].x;
			b.y = m_frame_list[i].vertex[m_index_list[j].b].z;
			b.z = m_frame_list[i].vertex[m_index_list[j].b].y;

			c.x = m_frame_list[i].vertex[m_index_list[j].c].x;
			c.y = m_frame_list[i].vertex[m_index_list[j].c].z;
			c.z = m_frame_list[i].vertex[m_index_list[j].c].y;

			a=a-c;
			c=b-c;
			b=a*c;
			b.norm();			

			// .a 's are now .b 's that's the difference
			*vPtr = m_frame_list[i].vertex[m_index_list[j].c].x; vPtr++;
			*vPtr = m_frame_list[i].vertex[m_index_list[j].c].z; vPtr++;
			*vPtr = m_frame_list[i].vertex[m_index_list[j].c].y; vPtr++;

			*vPtr = b.x; vPtr++;
			*vPtr = b.y; vPtr++;
			*vPtr = b.z; vPtr++;
            
			*vPtr = m_index_list[j].c_s; vPtr++;
			*vPtr = m_index_list[j].c_t; vPtr++;

			*vPtr = m_frame_list[i].vertex[m_index_list[j].b].x; vPtr++;
			*vPtr = m_frame_list[i].vertex[m_index_list[j].b].z; vPtr++;
			*vPtr = m_frame_list[i].vertex[m_index_list[j].b].y; vPtr++;

			*vPtr = b.x; vPtr++;
			*vPtr = b.y; vPtr++;
			*vPtr = b.z; vPtr++;

			*vPtr = m_index_list[j].b_s; vPtr++;
			*vPtr = m_index_list[j].b_t; vPtr++;

			*vPtr = m_frame_list[i].vertex[m_index_list[j].a].x; vPtr++;
			*vPtr = m_frame_list[i].vertex[m_index_list[j].a].z; vPtr++;
			*vPtr = m_frame_list[i].vertex[m_index_list[j].a].y; vPtr++;

			*vPtr = b.x; vPtr++;
			*vPtr = b.y; vPtr++;
			*vPtr = b.z; vPtr++;

			*vPtr = m_index_list[j].a_s; vPtr++;
			*vPtr = m_index_list[j].a_t; vPtr++;
		}

	}

	//cleanup as they are no longer needed
	for( i = 0; i < modelheader.num_frames; i++)
		n_delete_array (m_frame_list[i].vertex);

	n_delete_array(m_index_list);
	n_delete_array(m_frame_list);

	m_frame_list = NULL;
	m_index_list = NULL;
	
	return true;
}

//------------------------------------------------------------------------------
/**
*/
inline
bool
nMD2Loader::ReadIndices(void* buffer, unsigned int bufferSize)
{
    n_assert(buffer);
//    n_assert(bufferSize);

	if(Index16 == this->indexType)
	{
		n_assert(bufferSize == (this->numIndices * sizeof(ushort)));
        ushort* indexBuffer16 = (ushort*) buffer;

		// changed for proper group handling in meshbuilder
		// normally we only need one index (the one of the first group)
		for (int i=0; i<numIndices;i+=3)//(modelheader.num_tris * 3 * modelheader.num_frames); i+=3) 
		{
			indexBuffer16[i]=i;
			indexBuffer16[i+1]=i+1;
			indexBuffer16[i+2]=i+2;
		}
	}
	else
	{
		n_assert(bufferSize == (this->numIndices * sizeof(uint)));
        uint* indexBuffer32 = (uint*) buffer;

		// changed for proper group handling in meshbuilder
		// normally we only need one index (the one of the first group)
		for (int i=0; i<numIndices;i+=3)//(modelheader.num_tris * 3 * modelheader.num_frames); i+=3) 
		{
			indexBuffer32[i]=i;
			indexBuffer32[i+1]=i+1;
			indexBuffer32[i+2]=i+2;
		}
	}

    return true;
}

//------------------------------------------------------------------------------
/**
    A edge has the size of 4 * ushort, so you have to provide a buffer with the
    size numEdges * 4 * sizeof(ushort).
    The edge data is: ushort vertexIndex1, vertexIndex2, faceIndex1, faceIndex2;
    If a face Indicie is invalid (a border edge with only on face connected)
    the value is nMeshBuilder::InvalidIndex ( == -1).
*/
inline
bool
nMD2Loader::ReadEdges(void* buffer, unsigned int bufferSize)
{
    return false;
}


//------------------------------------------------------------------------------
/**
    BlendAnimations store the information which frames in the array belong to the
	same animation cycle. Array will be allocated and filled so clean it up !
	@param anims		[out]	adress to array of nBlendAnimations
	@param animCount	[out]	number of anims in array
*/
inline 
bool nMD2Loader::ReadAnimations(nBlendAnimations*anims, unsigned int animCount)
{
	n_assert(anims != NULL && animCount == this->m_anims);
	memcpy(anims,Anims,sizeof(nBlendAnimations)*animCount);
	return true;
}

inline
int nMD2Loader::GetAnimationCount()
{
	return this->m_anims;
}
/*
inline 
int nMD2Loader::GetVertexComponents() const
{
	return nMesh2::Coord|nMesh2::Normal|nMesh2::Texture;
}

inline 
int nMD2Loader::GetVertexWidth() const
{
	return 8;//v,t,n
}
*/
//------------------------------------------------------------------------------
#endif

