#ifndef N_BLENDMESHNODE_H
#define N_BLENDMESHNODE_H

#include "scene/nmaterialnode.h"
#include "anim/3di_nblendanimloader.h"
#include "anim/3di_md2loader.h"
#include "gfx2/ndynamicmesh.h"

/**
	@class nBlendMeshNode
	@ingroup 3DInfernoMD2ipol

	@brief scene node for MD2 models

*/

class nBlendMeshNode : public nMaterialNode
{
public:
	    /// constructor
    nBlendMeshNode();
    /// destructor
    virtual ~nBlendMeshNode();
    /// object persistency
    virtual bool SaveCmds(nPersistServer *ps);
    /// load resources
    virtual bool LoadResources();
    /// unload resources
    virtual void UnloadResources();

    /// indicate to scene server that we offer geometry for rendering
    virtual bool HasGeometry() const;
    /// render geometry
    virtual bool RenderGeometry(nSceneServer* sceneServer, nRenderContext* renderContext);
    /// get the mesh usage flags required by this shape node
    int GetMeshUsage() const;
    /// override the default mesh usage for this shape node
    void SetMeshUsage(int);

    /// set the mesh resource name
    void SetMesh(const char* name);
    /// get the mesh resource name
    const char* GetMesh() const;
    /// set the mesh group index
    void SetGroupIndex(int i);
    /// get the mesh group index
    int GetGroupIndex() const;
    /// set the NOH path to the mesh's resource loader
    void SetMeshResourceLoader(const char* resourceLoaderPath);
    /// get the mesh's resource loader
    const char* GetMeshResourceLoader();
	
    /// set current animation
    void SetAnimation(const char  *);
    /// get current animation
    const char * GetAnimation();
	/// get animation at index i (starting from 1)
    const char * GetAnimation(int i);
	/// get number of animations
	int GetAnimationCount();
	/// set numer of frames per second to be played
	void SetFPS(int);
	/// get number of frames per second to be played
	int GetFPS();
	/// time scale for interpolation
	float GetScale();
	/// time scale for interpolation
	void SetScale(float s);
	/// LOD for interpolation so far -1 not rendering, 0 is no interpolation, 1 is lerp interpolation
	int GetIpolLOD();
	/// LOD for interpolation so far -1 not rendering, 0 is no interpolation, 1 is lerp interpolation
	void SetIpolLOD(int i);
	/// Gets the loop type for animations
	int GetLoopType(void);
	const char *GetLoopName(void);
	/// Sets the loop type for animations
	void SetLoopType(int i);
	void SetLoopName(const char * c);

protected:
    /// load mesh resource
    bool LoadMesh();
    /// unload mesh resource
    void UnloadMesh();
	/// load from 3di mesh files
	bool LoadFile(nBlendAnimLoader *ml);
	/// interpolate between frames based on set LOD and Scale
	void Interpolate(nSceneServer* sceneServer, nRenderContext* renderContext);
	/// render mesh frame
	void RenderFrame(nSceneServer* sceneServer, nRenderContext* renderContext, int frameA);
	/// render mesh @todo use vertex tweening in directx
	void RenderLerp(nSceneServer* sceneServer, nRenderContext* renderContext, float lerp, int frameA, int frameB);
	/// return loop type from string
	int getLoopType(char * t);
	/// return loop type as string
	const char * getLoopType(int i);

    nRef<nMesh2> refMesh;
	nDynamicMesh dynMesh;
    nString meshName;
    int groupIndex;
    nDynAutoRef<nResourceLoader> refMeshResourceLoader;
    int meshUsage;

	int ipolLOD;
	float scale;
	/// @todo move this to nBlendAnimations as the interpolation type belongs to the animation
	int repType;
	
	enum N_IPOL
	{
		N_IPOL_IGNORE = -1,
		N_IPOL_FRAMES = 0,
		N_IPOL_LERP
	};
	enum N_REPTYPE
	{
		N_REPTYPE_LOOP =0,
		N_REPTYPE_ONESHOOT
	};


	/// the different animations in the md2 file (run, stand, attack1 etc.)
	nBlendAnimations		*Anims;
	/// number of different animations
	int						m_anims;	
	nBlendAnimations		*m_pLastAnim;
	int						m_LastFrame;
	nBlendAnimations		*m_pCurrAnim;
	int						m_CurrFrame;
	int						m_FPS;
	float					m_invFPS;
	float					m_AnimStart;
	int						m_Interpolator;
};


//------------------------------------------------------------------------------
/**
*/
inline
void
nBlendMeshNode::SetGroupIndex(int i)
{
    this->groupIndex = i;
}

//------------------------------------------------------------------------------
/**
*/
inline
int
nBlendMeshNode::GetGroupIndex() const
{
    return this->groupIndex;
}


//------------------------------------------------------------------------------
/**
    This method must return the mesh usage flag combination required by
    this shape node class. Subclasses should call SetMeshUsage in their
    constructors to ensure that this method reflects their requirements.

    @return     a combination on nMesh2::Usage flags
*/
inline 
int
nBlendMeshNode::GetMeshUsage() const
{
    return meshUsage;
}
 
//------------------------------------------------------------------------------
/**
    Specifies the mesh usage flag combination required by
    this shape node class.
*/
inline
void
nBlendMeshNode::SetMeshUsage(int usage)
{
    meshUsage = usage;
}



//------------------------------------------------------------------------------
/**
    Get name of current animation

    @return     name of current animation
*/
inline
const char *
nBlendMeshNode::GetAnimation()
{
	if(this->m_pCurrAnim)
		return this->m_pCurrAnim->name;
	else
		return NULL;
}

//------------------------------------------------------------------------------
/**
    Get count of total animation

    @return     number of animations
*/
inline
int
nBlendMeshNode::GetAnimationCount()
{
    return this->m_anims;
}

//------------------------------------------------------------------------------
/**
    Get name of animation i

    @return     index of animation wanted
*/
inline
const char *
nBlendMeshNode::GetAnimation(int i)
{
	if(this->Anims)
	{
		n_assert(i>0 && i < (this->m_anims +1))
		return this->Anims[i - 1].name;
	}
	else
		return NULL;
}

//------------------------------------------------------------------------------
/**
    Get number for fps at which the animations will be played

    @return     fps
*/
inline
int
nBlendMeshNode::GetFPS()
{
    return this->m_FPS;
}

//------------------------------------------------------------------------------
/**
    Set number for fps at which the animations will be played

    @param     fps
*/
inline
void
nBlendMeshNode::SetFPS(int i)
{
	n_assert(i > 0)
    this->m_FPS = i;
	this->m_invFPS = 1.0f/m_FPS;
}

//------------------------------------------------------------------------------
/**
    Set scale factor for animations played

    @param     f	scale factor animation will be scaled by
*/
inline
void
nBlendMeshNode::SetScale(float f)
{
	this->scale = f;
}

/**
    Get scale factor for animations played
*/
inline
float
nBlendMeshNode::GetScale(void)
{
	return this->scale;
}

//------------------------------------------------------------------------------
/**
    Set interpolation LOD -1 is no rendering at all, 0 is only raw frames will
	be rendered without interpolation, 1 means lerp interpolation will be used

    @param     i interpolation mode
*/
inline
void
nBlendMeshNode::SetIpolLOD(int i)
{
	switch(i)
	{
	case	N_IPOL_IGNORE:
	case	N_IPOL_FRAMES:
	case	N_IPOL_LERP:
			this->ipolLOD = i;
			return;
	}

	ipolLOD = N_IPOL_IGNORE;
}

//------------------------------------------------------------------------------
/**
    Get interpolation LOD -1 is no rendering at all, 0 is only raw frames will
	be rendered without interpolation, 1 means lerp interpolation will be used
*/
inline
int
nBlendMeshNode::GetIpolLOD(void)
{
	return ipolLOD;
}


//------------------------------------------------------------------------------
/**
    Sets loop type of ALL animations. 0 is loop, 1 is oneshoot

    @param     i loop type to be used
*/
inline
void
nBlendMeshNode::SetLoopType(int i)
{
	switch(i)
	{
	case N_REPTYPE_ONESHOOT: repType = N_REPTYPE_ONESHOOT; return;
	}

	repType = N_REPTYPE_LOOP;
}


//------------------------------------------------------------------------------
/**
    Sets loop type of ALL animations.  loop OR oneshoot

    @param     c loop type to be used
*/
inline
void
nBlendMeshNode::SetLoopName(const char * c)
{
	if(strcmp(c,"loop") != 0)
	{
		repType = N_REPTYPE_ONESHOOT;
	}
	else
	{
		repType = N_REPTYPE_LOOP;
	}
}


//------------------------------------------------------------------------------
/**
    Gets loop type of ALL animations. 0 is loop, 1 is oneshoot
*/
inline
int
nBlendMeshNode::GetLoopType(void)
{
	return repType;
}

inline
const char *
nBlendMeshNode::GetLoopName(void)
{
	if(repType != N_REPTYPE_ONESHOOT)
		return "loop";
	else
		return "oneshot";
}

#endif

