#ifndef N_NODLOADER_H
#define N_NODLOADER_H
//------------------------------------------------------------------------------
/**
    @class nNODLoader
	@ingroup 3DInfernoNODipol

    Loads a nod (vampire file format) mesh into user provided vertex and index buffers.
    
    (C) 2004 tom@3D-Inferno.com
*/
#include "gfx2/nmeshloader.h"
#include "gfx2/nmesh2.h"
#include "kernel/nfile.h"
//#include "anim/3di_nblendanimloader.h"
#include "anim/3di_nod.h"

//------------------------------------------------------------------------------
class nNODLoader : public nMeshLoader
{
public:
    /// constructor
    nNODLoader();
    /// destructor
    ~nNODLoader();
    /// open file and read header data
    virtual bool Open(nFileServer2* fs);
    /// close the file
    virtual void Close();
    /// read vertex data
    virtual bool ReadVertices(void* buffer, unsigned int bufferSize);
    /// read index data
    virtual bool ReadIndices(void* buffer, unsigned int bufferSize);
    /// read edge data
    virtual bool ReadEdges(void* buffer, unsigned int bufferSize);

private:
	NOD_prefix		prefix;
	NOD_Header		header;
	NOD_Material	*materials;
	NOD_bones		*bones;
	NOD_Mesh		*meshes;
	NOD_vertex		*vertices;
	NOD_LODinfo		*lod;
	NOD_face		*faces;
	NOD_meshgroup	*meshgroups;
};

//------------------------------------------------------------------------------
/**
*/
inline
nNODLoader::nNODLoader()
:bones(NULL),
meshes(NULL),
vertices(NULL),
lod(NULL),
faces(NULL),
meshgroups(NULL),
materials(NULL)
{

}

//------------------------------------------------------------------------------
/**
*/
inline
nNODLoader::~nNODLoader()
{
	if(bones)
		n_delete_array(bones);
	if(meshes)
		n_delete_array(meshes);
	if(vertices)
		n_delete_array(vertices);
	if(lod)
		n_delete_array(lod);
	if(faces)
		n_delete_array(faces);
	if(meshgroups)
		n_delete_array(meshgroups);
	if(materials)
		n_delete_array(materials);
}

//------------------------------------------------------------------------------
/**
*/
inline
bool
nNODLoader::Open(nFileServer2* fs)
{
    n_assert(!this->file);
    n_assert(fs);

    this->fileServer = fs;
    this->file = this->fileServer->NewFileObject();
    n_assert(this->file);

    // open the file
    if (!this->file->Open(this->filename.Get(), "r"))
    {
        n_printf("nNODLoader: could not open file '%s'!\n", this->filename.Get());
        this->Close();
        return false;
    }


    // read header
	this->file->Read(&prefix, sizeof(NOD_prefix));
	if(prefix.version != 7) // IDP2 not a quake 2 file
	{
		n_printf("File <%s> is not in NOD format.",this->filename.Get());
		this->file->Close();
		return false;
	}
	//allocate space for materialnames
	materials = n_new(NOD_Material[prefix.materials]);
	//read materials
	file->Read(&materials, sizeof(NOD_Material)*prefix.materials);
	//read header
	file->Read(&header,sizeof(NOD_Header));

	bones = n_new(NOD_bones[header.bones]);
	meshes = n_new(NOD_Mesh[header.meshes]);
	vertices = n_new(NOD_vertex[header.verts]);
	if(header.flags & 0x1)//LOD bit set ?
	{
		lod = n_new(NOD_LODinfo[header.verts]);
	}
	faces = n_new(NOD_face[header.faces]);
	meshgroups = n_new(NOD_meshgroup[header.meshgroups]);

	this->numGroups = header.meshgroups;
	this->numEdges = 0;
	this->numIndices = header.faces * 3;
	this->numTriangles = header.faces;
	this->numVertices = header.verts;

	this->vertexComponents = nMesh2::Coord|nMesh2::Normal|nMesh2::Texture;
	this->vertexWidth = 8;

    return true;
}

//------------------------------------------------------------------------------
/**
*/
inline
void
nNODLoader::Close()
{
    if (this->file)
    {
        if (this->file->IsOpen())
        {
            this->file->Close();
        }
        this->file->Release();
        this->file = 0;
    }
}

//------------------------------------------------------------------------------
/**
*/
inline
bool
nNODLoader::ReadVertices(void* buffer, unsigned int bufferSize)
{
    //n_assert(buffer);

	//Read bones array
	file->Read(bones,sizeof(NOD_bones)*header.bones);

	//read meshdefs (only names)
	file->Read(meshes,sizeof(NOD_Mesh)*header.meshes);

	//read vertices 
	file->Read(vertices, sizeof(NOD_vertex)*header.verts);

	//lod data
	if(header.flags & 0x1)//LOD bit set ?
	{
		file->Read(lod, sizeof(NOD_LODinfo)*header.verts);
	}

	//read faces
	file->Read(faces,sizeof(NOD_face)*header.faces);

	//read meshgroups
	file->Read(meshgroups, sizeof(NOD_meshgroup)*header.meshgroups);


	//each meshgroup its own meshgroup :-)
	int vert = 0;
	int ind = 0;
	for(int i = 0; i < header.meshgroups;i++)
	{
		nMeshGroup meshGroup;
		meshGroup.SetFirstVertex(vert);
		vert += meshgroups[i].vertices;
		meshGroup.SetNumVertices(meshgroups[i].vertices);
		meshGroup.SetFirstIndex(ind);
		ind += meshgroups[i].faces *3;
		meshGroup.SetNumIndices(meshgroups[i].faces *3);
		meshGroup.SetFirstEdge(0);
		meshGroup.SetNumEdges(0);
		this->groupArray.Append(meshGroup);
	}
	return true;
}

//------------------------------------------------------------------------------
/**
*/
inline
bool
nNODLoader::ReadIndices(void* buffer, unsigned int bufferSize)
{
    n_assert(buffer);

	if(Index16 == this->indexType)
	{
		n_assert(bufferSize == (this->numIndices * sizeof(ushort)));
        ushort* indexBuffer16 = (ushort*) buffer;

		// changed for proper group handling in meshbuilder
		// normally we only need one index (the one of the first group)
		for (int i=0; i<numIndices;i+=3)//(modelheader.num_tris * 3 * modelheader.num_frames); i+=3) 
		{
			indexBuffer16[i]=i;
			indexBuffer16[i+1]=i+1;
			indexBuffer16[i+2]=i+2;
		}
	}
	else
	{
		n_assert(bufferSize == (this->numIndices * sizeof(uint)));
        uint* indexBuffer32 = (uint*) buffer;

		// changed for proper group handling in meshbuilder
		// normally we only need one index (the one of the first group)
		for (int i=0; i<numIndices;i+=3)//(modelheader.num_tris * 3 * modelheader.num_frames); i+=3) 
		{
			indexBuffer32[i]=i;
			indexBuffer32[i+1]=i+1;
			indexBuffer32[i+2]=i+2;
		}
	}

    return true;
}

//------------------------------------------------------------------------------
/**
    A edge has the size of 4 * ushort, so you have to provide a buffer with the
    size numEdges * 4 * sizeof(ushort).
    The edge data is: ushort vertexIndex1, vertexIndex2, faceIndex1, faceIndex2;
    If a face Indicie is invalid (a border edge with only on face connected)
    the value is nMeshBuilder::InvalidIndex ( == -1).
*/
inline
bool
nNODLoader::ReadEdges(void* buffer, unsigned int bufferSize)
{
    return false;
}


//------------------------------------------------------------------------------
#endif

