//------------------------------------------------------------------------------
// (c) 01-2002 Gottfried Chen
//------------------------------------------------------------------------------

#ifndef STATISTICAL_WATER_H
#define STATISTICAL_WATER_H

//#pragma once

//#include "mathlib/nmath.h"
//#include "water/types.h"
#include "mathlib/vector.h"
#include "water/complex.h"
#include "water/fft.h"
//#include "kernel/ntypes.h"

//class Fft2D;
//#define n_log(x) log(x)

const float G = 9.81f;
const float INV_SQRT_TWO(0.70710678118654752440084436210485f);

//moved to ncomplex
//inline nComplex conj(nComplex& _X)
//{return nComplex (_X.real(), -(_X.imag()));}
// Heightmap:
// x is right
// y is up
class StatisticalWater
{
public:
    // Parameters:
    // - <x>, <y> are the sizes in world units (i.e. meters).
    // - 2^<rowPower>, 2^<columnPower> are the number of rows, columns in
    //   the array used to represent the water.
    // - <a> is a numerical constant used for the Phillips spectrum.
    // - <wind> is a vector in world units describing the wind direction
    //   and speed.
    //
    // Reasonable values are:
    // (6, 6, 0.0008f, Vector2(30,10), 100.0f, 100.0f);
    StatisticalWater(unsigned int xPower, unsigned int yPower,
                     float a, const vector2& wind,
                     float xWorld, float yWorld);
    ~StatisticalWater();

    // Time is absolute (i.e. not relative to last frame)
    void update(const double&);

    const nComplex* getMap() const;
    unsigned int getXSize() const;
    unsigned int getYSize() const;

protected:
    unsigned int getXPower() const;
    unsigned int getYPower() const;

    // The tick() function calls store() in a loop over the whole grid.
    // tick() only calculates the h~(K, t) value from the Tessendorf paper.
    // store() stores the value into the frequency domain map for later
    // transformation. ChoppyWaves additionally stores values for the choppy
    // delta map.
    // <index> is the current index in mh.
    // <result> is h~(K, t)
    // <k> is K
    // <kLen> is <k>.length()
    virtual
    void store(unsigned int index, const vector2& k, float kLen, const nComplex& result);

    // After calculating the heightfield in the frequency domain. Transform
    // it back to the time domain.
    virtual
    void transformToTime();

    const Fft2D* getFft() const;

private:
    void calculateH0();
    // Transforms an array index into a world coordinate. (The array has a
    // [mRows][mColumns] index range. But the grid is centered at 0).
    float indexToWorldX(unsigned int index);
    float indexToWorldY(unsigned int index);

    nComplex* mH0;
    nComplex* mh;

    float mA;
    vector2 mWind;
    float m2PiByXWorld;
    float m2PiByYWorld;
    unsigned int mXPower;
    unsigned int mYPower;
    unsigned int mXSize;
    unsigned int mYSize;
    Fft2D *mFft;
};

//------------------------------------------------------------------------------
inline unsigned int StatisticalWater::getXPower() const
//------------------------------------------------------------------------------
{
    return mXPower;
}

//------------------------------------------------------------------------------
inline unsigned int StatisticalWater::getYPower() const
//------------------------------------------------------------------------------
{
    return mYPower;
}

//------------------------------------------------------------------------------
inline float StatisticalWater::indexToWorldX(unsigned int index)
//------------------------------------------------------------------------------
{
    return (float(index) - mXSize/2)*(m2PiByXWorld);
}

//------------------------------------------------------------------------------
inline float StatisticalWater::indexToWorldY(unsigned int index)
//------------------------------------------------------------------------------
{
    return (float(index) - mYSize/2)*(m2PiByYWorld);
}


//------------------------------------------------------------------------------
// Helpers
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
inline int neg1Pow(int n)
//------------------------------------------------------------------------------
{
    static int pow[2] = { 1, -1 };
    return pow[n&1];
}

//------------------------------------------------------------------------------
inline int mod(int a, unsigned int b)
//------------------------------------------------------------------------------
{
    return a&(b-1);
}

//------------------------------------------------------------------------------
inline float frand(float min, float max)
//------------------------------------------------------------------------------
{
    return min+(float(::rand())/float(RAND_MAX))*(max-min);
}

//------------------------------------------------------------------------------
inline int rand(int min, int max)
//------------------------------------------------------------------------------
{
    return min + (::rand()%(max-min+1));
}

//------------------------------------------------------------------------------
inline float gaussian()
//------------------------------------------------------------------------------
{
    static float y1, y2;
    static bool calculate(true);
    float x1, x2, w;

    if (calculate)
    {
        calculate = false;

        do
        {
            x1 = 2.0f*frand(0, 1) - 1.0f;
            x2 = 2.0f*frand(0, 1) - 1.0f;
            w = x1*x1 + x2*x2;
        } while (w >= 1.0f);

        w = sqrt((-2.0f * log(w))/w);
        y1 = x1*w;
        y2 = x2*w;
        
        return y1;
    }
    else
    {
        calculate = true;
        return y2;
    }
}

#define		sqr(x)			((x)*(x))	

//------------------------------------------------------------------------------
//float phillips(float a, const vector2& k, const vector2& wind)
inline float phillips(float a, vector2& k, vector2& wind)
//------------------------------------------------------------------------------
{
    //float k2 = k.lengthSquare();
	float k2 = sqr(k.len());
    if (k2 == 0)
        return 0;

    //float v2 = wind.lengthSquare();
	float v2 = sqr(wind.len());
    float l = v2/G;

    float ret = a*
                (exp(-1/(k2*sqr(l))) / sqr(k2))*
//                (sqr(k*wind) / (k2*v2));//   TMI implemented dotproduct on operator %
                (sqr(k%wind) / (k2*v2));

  //  GE_ASSERT_MSG(ret>=0, "We take the sqrt of this value later, so it must be positive");
    return ret;
}

#endif // STATISTICAL_WATER_H