#ifndef N_WATERNODE_H
#define N_WATERNODE_H


//------------------------------------------------------------------------------
///helper function to compute the power of 2 of given x
inline int PowerOf2(int pow)
{
	int erg=0;
	while(pow > 1)
	{
		erg++;
		pow=pow>>1;
	}
	return pow<<erg;
}

/**
    @class nWaterNode
	@ingroup 3DInfernoWater  
    @brief Scene node providing water meshes

    A nWaterNode is the scene representatio of a certain water algorithm.
	At the moment there are three water algorithms available:<br>
	-Gerstner waves
	-Choppy waves
	-Statistical waves

	A nWaterNode uses the meshes created by nWaterServer for rendering. During
	Rendering update() on the nWaterServer is called for the base sample to be
	updated. The node just uses the mesh provided by the server for rendering.
	A nWaterNode in this design is just a render instance of a foreign mesh.
	Height queries are passed over to the waterserver to retrieve values for the
	set algorithm. 
	By default the node starts out with statistical water and the only information
	stored is the algorithm used.

	(C)	2002-2004	tom@3d-inferno.com
*/

#include "scene/nmaterialnode.h"
#include "gfx2/nmesh2.h"
#include "water/nwaterserver.h"


//------------------------------------------------------------------------------
class nWaterNode : public nMaterialNode
{
public:
    /// constructor
    nWaterNode();
    /// destructor
    virtual ~nWaterNode();
    /// object persistency
    virtual bool SaveCmds(nPersistServer *ps);
    /// load resources
    virtual bool LoadResources();
    /// unload resources
    virtual void UnloadResources();

    /// indicate to scene server that we offer geometry for rendering
    virtual bool HasGeometry() const;
    /// render geometry
    virtual bool RenderGeometry(nSceneServer* sceneServer, nRenderContext* renderContext);

    /// get the mesh usage flags required by this shape node
    int GetMeshUsage() const;

	/// sets the type of water algorithms used. 0: Statistical 1:Choppy 2:Gerstner
	bool  SetType(int _t);
	/// returns the type of water algorithm the node uses
	int GetType();
	//inline int GetSize(){ return this->m_Size;};
	/// returns the height at a given psition
	float GetHeight(float i, float j);
	//void SetSize(int i);
protected:
//    nAutoRef<nGfxServer2> refGfx2;
	//int m_Size;
	//int m_NSize; //new x
	/// type of algorithms used
	int m_Type;
	/// new type of algorithm to be used
	int m_Nt;
	//bool m_Render;
	/// reference to mesh filled by nWaterServer
	nRef <nMesh2> ref_vb;
	nAutoRef<nWaterServer> ref_Server;
	//bool initBuffer();
	/// lookup of mesh created by nWaterServer
	bool Load();
};


//------------------------------------------------------------------------------
/**
    Initializes the start type to statistical water and registers interrest in
	the desired algorithm in the nWaterServer
*/
inline nWaterNode::nWaterNode():
	//m_Size(SampleSize),//SampleSize),
	//m_NSize(SampleSize),
	m_Type(nWaterServer::N_STATISTIC),
	m_Nt(nWaterServer::N_STATISTIC)
//	m_Render(false)
{
    ref_Server = "/sys/servers/water";
	if(ref_Server.isvalid())
		ref_Server->AddWaterRef(m_Type);

//	m_Size=PowerOf2(SampleSize);
}

//------------------------------------------------------------------------------
/**
    Removes interrest for algorithm in nWaterServer and releases resources
*/
inline nWaterNode::~nWaterNode()
{
	if(this->ref_Server.isvalid())
	{
		ref_Server->RemWaterRef(this->m_Type);
		this->ref_Server.invalidate();
	}

	//release mesh reference and materials
	UnloadResources();
}

//------------------------------------------------------------------------------
/**
    Returns the height at given position. i,j should be in local coordinates.

	@todo take care of values passed in in world coordinates

    @param  i      X Position
	@param  j      Z Position

    @return        Y Position for passed parameters
*/
inline float nWaterNode::GetHeight(float i, float j)
{
	if(ref_Server.isvalid())
	{
		return ref_Server->GetHeight(i,j,m_Type);
	}
	return 0.0f;
}

//------------------------------------------------------------------------------
/**
    Set the type of water algorithm to be used. 0 Stat , 1 Chop, 2 Gerstner @see
	nWaterServer

    @param  i      type of algorithm used 0 Stat , 1 Chop, 2 Gerstner
    @return        true if success
*/
inline bool nWaterNode::SetType(int i)
{
	if(i < 0 || i> 2)//3) three not supported
		return false;
	else
		this->m_Nt = i;
	return true;
}

//------------------------------------------------------------------------------
/**
    Returns the actually set water algorithm for this node

    @return                type of algorithm used
*/
inline int nWaterNode::GetType()
{
    return this->m_Type;
}


//------------------------------------------------------------------------------
/**
    Return Mesh usage flags from server

	@return		MeshUsage flags

*/
inline int nWaterNode::GetMeshUsage() const
{
	return VertType;
}


//------------------------------------------------------------------------------
/**
    Return true as we provide a mesh for rendering

*/
inline bool nWaterNode::HasGeometry() const
{
	return true;
}

#endif

