#ifndef N_WATERSERVER_H
#define N_WATERSERVER_H
//--------------------------------------------------------------------
/**
    @class nWaterServer
	@ingroup 3DInfernoWater  
    @brief Server for providing different water algorithms

    The water server can be found at "/sys/servers/water". It provides one
	place for all water algorithms. At the moment there are three water algorithms
	available:<br>
	<ul>
	<li>Gerstner waves</li>
	<li>Choppy waves</li>
	<li>Statistical waves</li>

	The server is a simplified version of the old water algorithms server.
	The server registers references to water algorithms placed by nWaterNodes.
	It is possible to have different water algorithms running in parallel.
	Each water algorithm has an instance which holds a base sample of height
	data. From this sample the final mesh is created. In this version the 
	sample size and the size of the final mesh to be rendered are the same.
	A nWaterNode calls Update() on the waterserver which then starts an update
	cycle. On each frame during an update cycle a different algorithm is updated and
	evetually the mesh holding the final render data is updated too. After all
	algorithms are updated and all meshes of used algorithms are actuallized the
	update cycle is finished. A new update cycle will start after predefined
	frame time. During this time all nWaterNodes use the mesh instances created
	in the last update cycle for rendering.
	nWaterNodes do not hold any mesh data, or any mesh logic, as it is all centralized 
	in the nWaterServer.

	As all algorithms are diagonal symetric in their height data nWaterNodes
	can be tiled.

    - Not finally ported from Nebula1
	An additional mechanism can be used to thin out the neccessary meshes for
	rendering. You can create a field of 3*3 nWaterNodes of the same size.
	Then place a 3to1 bridge nWaterNode around them. By doing so you can now attach
	one nWaterNode scaled to 3 times the normal size to the former 3*3 field.
	
	(C)	2002-2004	tom@3d-inferno.com
*/
//--------------------------------------------------------------------


#include "kernel/nroot.h"
#include "kernel/nautoref.h"
#include "gfx2/nmesh2.h"
#include "water/gerstnerwaves.h"
#include "water/choppywaves.h"

//--------------------------------------------------------------------
/// Size of meshes and samples
const int SampleSize = 32;
/// @todo nMesh2::WriteOnce works at the moment but is an abuse of write once as we update it!
const int VertType = (nMesh2::Coord|nMesh2::Uv0|nMesh2::Normal);//nMesh2::Color|
class nWaterServer : public nRoot {
public:

	/// available water types
	enum N_WATER
	{
		N_STATISTIC = 0,
		N_CHOPPY,
		N_GERSTNER,
		N_CHOPPY_3TO1
	};

	/// constructor
	nWaterServer();
	/// destructor
	~nWaterServer();
	/// save waterserver settings
	virtual bool SaveCmds(nPersistServer* fs);
	/// add interrest in certain water algorithm
	bool AddWaterRef(int);
	/// remove interrest in certain water algorithm
	bool RemWaterRef(int);
	/// returns mesh usage (always write once)
	const int GetMeshUsage(){ return nMesh2::WriteOnce;};
	/// set the update frequency in frames per second
	void SetFPS(int);
	/// returns the update frequency in frames per second
	int GetFPS();
	/// returns the height at a grid position for a certain water algorithm
	float GetHeight(int x, int z, int algo);
	/// returns the height above a grid
	float GetHeight(float x, float z, int algo);
	/// update water algorithms if needed
	bool Update();
	/// returns number of total water references
	int GetRefs();
	/// returns the choppyness for the choppy water algorithm
	float GetChopp();
	/// sets the choppyness for the choppy water algorithm
	void SetChopp(float _t);
	/// Sets creation and usage of 3 to 1 bridge
	void Create3to1Mesh(bool f);
private:
	/// returns complex height array valid for statistical and choppy water
	const nComplex *getCMap();
	/// returns complex X deviation array valid only for choppy water
	const nComplex *getXMap();
	/// returns complex Z deviation array valid only for choppy water
	const nComplex *getZMap();
	/// returns vector array valid only for gerstner water
	vector3 *getGMap();
	/// updates choppy water mesh
	void updateChoppy(void);
	/// updates statistical water mesh
	void updateStatistic(void);
	/// updates gerstner water mesh
	void updateGerstner(void);
	/// update 3 to 1 bridge for a given algorithm
	void update3to1(int);
	/// set mesh group information in given mesh
	void initGroup(nMesh2 * pmesh);
	/// base index fills ref_ibuf
	void initIndex(void);
	/// fill s_vb mesh with index data
	void initStatisticalIndex(void);
	/// fill c_vb mesh with index data
	void initChoppyIndex(void);
	/// fill g_vb mesh with index data
	void initGerstnerIndex(void);
	/// fill the bridge mesh with index data, not sed at the moment.
	void init3to1Index(void);
	/// Height in a statistical water field at given coordinates
	float GetSH(float x, float z);
	/// Height in a choppy water field at given coordinates
	float GetCH(float x, float z);
	/// Height in a gerstner water field at given coordinates
	float GetGH(float x, float z);
	/// choppy mesh
	nRef<nMesh2> c_vb;
	nRef<nMesh2> c_3to1vb;
	/// statistical mesh
	nRef<nMesh2> s_vb;
	nRef<nMesh2> s_3to1vb;
	/// gerstner mesh
	nRef<nMesh2> g_vb;
	nRef<nMesh2> g_3to1vb;
	/// one index array which is copied over to meshes
	ushort *ref_ibuf;
	/// one 3 to 1 bridge index copied over to 3 to 1 meshes
	ushort *ref_i3to1buf;
	/// statistical and choppy water instance
	ChoppyWaves *m_Choppy;
	/// gerstner water instance
	GerstnerWaves *m_Gerstner;
	//nAutoRef<nGfxServer2> refGfx;
	bool m_3to1Mesh;
	/// number of references to statistical water
	int m_A;
	/// number of references to choppy water
	int m_B;
	/// number of references to gerstner water
	int m_C;
	/// FPS for update of algorithms
	int m_FPS;
	/// 1/FPS
	float m_delta;
	/// last time the algorithms got updated (last update cycle was finished)
	float m_last;
	/// last index in update cycle
	int m_Updated;
};



//--------------------------------------------------------------------
#endif