#include "3di/3di_spherepack.h"
#include "3di/3di_spherepackfactory.h"
#include <memory.h> //for debug memset

SpherePack::SpherePack(void):mParent(NULL), mChilds(NULL), 
		mPrevSibling(NULL),
		mNextSibling(NULL),
		mToParent(0.0f),
		mChildCount(0),
		mUserData(NULL),
		mIntegrate(false),
		mRecompute(false),
		mFactory(NULL)
{};

SpherePack::~SpherePack(void)
{
	// we do not own the userdate, just null it, don't release it.
	if(mUserData)
	{
		mUserData = NULL;
	}
	Reset();
}

void SpherePack::Init(SpherePackFactory *factory, const vector3 &pos, 
					  float radius, void *userdata)
{
	mParent=NULL;
	mChilds=NULL; 
	mPrevSibling=NULL;
	mNextSibling=NULL;
	mToParent=0.0f;
	mChildCount=0;
	mUserData=userdata;
	mIntegrate=false;
	mRecompute=false;
	mFactory=factory;
	Set(pos,radius);
};



void SpherePack::NewPos(const vector3 &pos)
{
  mPosition = pos;

  // if we have a parent and we have not already been flagged for re-integration, then.....
  if ( mParent && !mIntegrate)
  {
    float dist = (mParent->GetPosition() - mPosition).lensquared();  // compute squared distance to our parent.
    if ( dist + this->GetSquaredRadius() > mParent->GetSquaredRadius()) // if that exceeds our binding distance...
    {
	  mFactory->AddRecompute(mParent);
      // Unlink ourselves from the parent sphere and place ourselves into the root node.
      Unlink();
      mFactory->AddIntegrate(this); // add ourselves to the re-integration fifo.
    }
  }
}

void SpherePack::NewPosRadius(const vector3 &pos,float radius)
{
  // New position and, possibly, a new radius.
  mPosition    = pos;

  if ( mParent && !mIntegrate)
  {
	if ( radius != GetRadius() )
	{
		SetRadius(radius);
	}

	float dist = (mParent->GetPosition() - mPosition).lensquared();  // compute squared distance to our parent.
	if ( dist + this->GetSquaredRadius() > mParent->GetSquaredRadius()) // if that exceeds our binding distance...
	{
		mFactory->AddRecompute(mParent);
		Unlink();
		mFactory->AddIntegrate(this);
	}
	else
	{
		mFactory->AddRecompute(mParent);
	}
  }  

}

/**
	Unlink from siblings, unlink in parent
*/
void SpherePack::Unlink(void)
{
	mIntegrate = false;
	mRecompute = false;

	//unlink us in the parent must always come before loosing info
	if(mParent)
	{mParent->LostChild(this);
	mParent = NULL;
	}//if we have siblings we unlink us from them
	if(mPrevSibling)
	{
		mPrevSibling->mNextSibling = this->mNextSibling;
	}
	if(mNextSibling)
	{
		mNextSibling->mPrevSibling = this->mPrevSibling;
	}
	this->mNextSibling = NULL;
	this->mPrevSibling = NULL;

};

/**
	Add as child, grow one and add as child or add to child as child 
*/
void SpherePack::AddChild(SpherePack *pack)
{
	// case a, as we have no children at all
	if( !mChilds )
	{
		addAsChild(pack);
		return;
	}
	
	float mindist=3e38f;
	float currdist=0.0f;
	SpherePack *nearest=NULL;
	SpherePack *current = mChilds;
	// walk all children
	while(current)
	{
		currdist = (current->mPosition - pack->mPosition).len();
		
		//completetly inside this one !
		if(pack->GetRadius() + currdist <= current->GetRadius())//this->GetRadius())
		{
			current->AddChild(pack);
			return;
		}
		//part in part out									
		else if(pack->GetRadius() + current->GetRadius() >= currdist)
		{
			//new nearest ?
			if(currdist < mindist)
			{
				mindist = currdist;
				nearest = current;
			}
		}
		////completely out of this node  so do nothing
		//else {}
		current = current->GetPrevSibling();
	}//end of walk all children

	//if we have a nearest then try to grow it
	if(nearest)
	{
		nearest->addGrowChild(pack,mindist);
	}
	//else we did not cut any child so add this one directly to childs
	else
		addAsChild(pack);
};

/**
	Just add it to our childs
	Fix linkage between children
*/
void SpherePack::addAsChild(SpherePack *pack)
{
	pack->SetParent(this);
	
	if(!mChilds)
	{
		pack->SetParent(this);
		this->mChilds = pack;
		this->mChildCount++;
		return;
	}
/*
if(child)
				{
					prev = child->prev;
					next = child;
					child->prev = this;
					if(prev)
						prev->next = this;
				}
*/
	pack->mPrevSibling = this->mChilds->mPrevSibling;
	pack->mNextSibling = this->mChilds;
	this->mChilds->mPrevSibling = pack;
	if(pack->mPrevSibling)
		pack->mPrevSibling->mNextSibling = pack;

	this->mChildCount++;
};

/**
	Try to grow but obey leaf size for leaf nodes
	If not possible to grow then create new supersphere to contain both
	Add pack to supersphere or grown self
*/
void SpherePack::addGrowChild(SpherePack *pack,float distance)
{
	// check if we blow max leaf size if so create new supersphere
	if(mChilds &&(distance > mFactory->GetMaxLeafSize()))
	{
		SpherePack *super = mFactory->AddSphere((mPosition-pack->GetPosition())*0.5f, distance*0.5f,NULL);
		super->SetParent(this->mParent);
		this->Unlink();
		super->addAsChild(this);
		pack->Unlink();
		super->addAsChild(pack);
		//mFactory->AddIntegrate(super);
		mFactory->AddRecompute(mParent);
	}
	// grow without pain, recomp parent thereafter as we are not a leaf node
	else //if(distance > mFactory->GetMaxLeafSize())
	{
		n_assert(distance < mFactory->GetMaxRootSize())
		//new pos middle between oldpos and pack
		//new radius = distance/2
		NewPosRadius(mPosition+((pack->GetPosition()-mPosition)*0.5f), distance*0.5f);
		addAsChild(pack);
		mFactory->AddRecompute(mParent);
	}
};


void SpherePack::Remove(void)
{
	//remove children
	SpherePack * child = mChilds;
	while(child)
	{
		child->Remove();
		child = child->GetNextSibling();
	}
	this->Reset();
}

void SpherePack::Reset(void)
{
	Unlink();
	this->mChilds=NULL; 
	this->mPrevSibling=NULL;
	this->mNextSibling=NULL;
	this->mToParent=0.0;
	this->mChildCount=0;
	if(this->mUserData)
		this->mUserData=NULL;
	this->mIntegrate=false;
	this->mRecompute=false;
}

/**
	recalc size and check for remove
*/
bool SpherePack::Recompute(float gravy)
{
	if(!mParent)
		return false; // this is the root node, it has no parent

	if(!mChilds && !mUserData)
		return true; // this is an empty node !

	// recompute bounding sphere!
	vector3 total(0,0,0);
	int count=0;
	SpherePack *pack = mChilds;
	while ( pack )
	{
		total+=pack->GetPosition();
		count++;
		pack = pack->GetPrevSibling();
	 }

  if ( count )
  {
	vector3 oldpos = mPosition;
   // float recip = 1.0f / float(count);
    mPosition = total/float(count);//*recip;
    

    //mPosition = total; // new origin!
    float maxradius = 0;

    pack = mChilds;

    while ( pack )
    {
      float dist = (pack->GetPosition() - this->GetPosition()).lensquared();
      float radius = sqrtf(dist) + pack->GetRadius();
      if ( radius > maxradius )
      {
        maxradius = radius;
      //  if ( (maxradius+gravy) >= GetRadius() )
      //  {
      //    mPosition = oldpos;
      //    return false;
      //  }
      }
      pack = pack->GetPrevSibling();
    }

    maxradius+=gravy;

    SetRadius(maxradius);


  }



	return false;
}

/**
	Decrement child count and fix our start child pointer if neccessary
*/
void SpherePack::LostChild(SpherePack *t)
{
	n_assert((this->mChildCount)>0)
	
	this->mChildCount--;

	// check if leader to children if so adjust leader
	if((this->mChilds)!= t)
		return;

	if(t->mPrevSibling)
	{
		this->mChilds = t->mPrevSibling;
		return;
	}

	if(t->mNextSibling)
	{
		this->mChilds = t->mNextSibling;
		return;
	}
	
	n_assert((this->mChildCount == 0))
	this->mChilds = NULL;
}

void SpherePack::PrintDebugInfo(int level)
{
	// for a fast debug just use an already spaced array 
	n_assert(level < 256);
	char pad[256]="";

	if(level >0)
	{
		memset(pad,' ',level); 
		pad[level] = '\0';
	}
	vector3 p = mPosition;
	n_printf("%sP%f.1 %f.1 %f.1  R%f.1\n",pad,p.x,p.y,p.z,GetRadius());
	SpherePack * c = mChilds;
	while(c)
	{
		c->PrintDebugInfo(level+1);
		c = c->GetPrevSibling();
	}
}