#include "anim/3di_nblendmeshnode.h"
#include "gfx2/nmesh2.h"
#include "gfx2/ngfxserver2.h"
#include "kernel/ntimeserver.h"

nNebulaScriptClass(nBlendMeshNode, "nmaterialnode");


nBlendMeshNode::nBlendMeshNode():
meshUsage(0),
Anims(NULL),
groupIndex(0),
meshName(NULL),
refMeshResourceLoader(NULL),
m_CurrFrame(NULL),
m_LastFrame(NULL),
m_pCurrAnim(NULL),
m_pLastAnim(NULL),
scale(1.0f),
m_FPS(1),
m_invFPS(1.0f),
repType(N_REPTYPE_LOOP),
ipolLOD(N_IPOL_FRAMES)
{

}

nBlendMeshNode::~nBlendMeshNode()
{
	UnloadMesh();
}

void nBlendMeshNode::UnloadMesh()
{
	if(refMesh.isvalid())
	{
		refMesh->Release();
	}
	if(Anims)
		n_delete_array(Anims);

	Anims = NULL;
	m_anims = m_LastFrame = m_CurrFrame = 0;
	m_AnimStart = 0.0f;
	groupIndex = 0;
	//meshName = NULL;
	meshUsage = nMesh2::ReadOnly;
}

//------------------------------------------------------------------------------
/**
    Load new mesh, release old one if valid. Also initializes the groupIndex
    member.
*/

bool
nBlendMeshNode::LoadMesh()
{
    if ((!this->refMesh.isvalid()) && (!this->meshName.IsEmpty()))
    {
        // append mesh usage to mesh resource name
        nString resourceName = this->meshName.Get();
        resourceName += "_";
        resourceName.AppendInt(this->GetMeshUsage());

        // get a new or shared mesh
        this->refMesh = nGfxServer2::Instance()->NewMesh(resourceName.Get());
        n_assert(this->refMesh);        
        if (!this->refMesh->IsValid())
        {
			nPathString filename(this->meshName.Get());

            // do not set this !
			//this->refMesh->SetFilename(this->meshName.Get());
			this->refMesh = nGfxServer2::Instance()->NewMesh(resourceName.Get());
			n_assert(refMesh);
			//this->refMesh->SetVertexComponents(VertType);
			//this->refMesh->SetUsage(this->GetMeshUsage());
			//creates empty buffers as the filename of the mesh was left out
			//will be called inside of load
			//this->refMesh->Load();


			nBlendAnimLoader* meshLoader = 0;

			// select meshloader
			if (filename.CheckExtension("md2") || filename.CheckExtension("mdl"))
			{
				meshLoader = n_new(nMD2Loader);
			}
			else
			{
				n_error("nBlendMeshNode::LoadResource: filetype not supported!\n");
			}

			if (0 != meshLoader)
			{
				meshLoader->SetFilename(meshName.Get());
				this->LoadFile(meshLoader);
				n_delete (meshLoader);
				// init to first animation stand or idel anim recommended
				this->m_pCurrAnim = this->Anims;
				this->m_pLastAnim = this->m_pCurrAnim;
				this->m_AnimStart = -1.0f;
				this->m_LastFrame = this->m_pCurrAnim->frames;
				this->m_CurrFrame = 1;
			}
        }
       // this->refMesh = mesh;
        this->SetLocalBox(this->refMesh->GetGroup(this->groupIndex).GetBoundingBox());
    }
    return true;
}

bool nBlendMeshNode::LoadFile(nBlendAnimLoader *ml)
{
	// hmmm, already loaded.  happens in the async case where a job
    // if removed from the queue and then immediately re-added to the queue
    // before it is unloaded.
    if (this->refMesh->IsValid())
        return true;

    n_assert(!refMesh->IsValid());

	if(!ml->Open(kernelServer->GetFileServer()))
	{
        n_error("nBlendMesh: could not open file '%s'!\n", refMesh->GetFilename().Get());
        return false;
	}

    // read in the vertex count
    unsigned int vertexcount = ml->GetNumVertices();

    // there are 4 shorts per vertex: x,y,z, and morph amount
    unsigned int valuecount = vertexcount * ml->GetVertexWidth();
    
	float * vertexbuffer = n_new (float[valuecount]);
    ml->ReadVertices(vertexbuffer, valuecount * sizeof(float) );

	unsigned int indexcount = ml->GetNumIndices();

	ushort * indexbuffer = n_new (ushort[indexcount]);
	ml->ReadIndices(indexbuffer,indexcount * sizeof(ushort));
	
	m_anims = ml->GetAnimationCount();
	Anims = n_new (nBlendAnimations[m_anims]);
	ml->ReadAnimations(Anims,m_anims);
	
	n_assert(refMesh.isvalid());

    refMesh->SetNumGroups(ml->GetNumGroups());
    refMesh->SetNumVertices(vertexcount);
    refMesh->SetNumIndices(indexcount);
	refMesh->SetUsage(nMesh2::ReadOnly);
    //refMesh->SetUsage(nMesh2::WriteOnce);
	refMesh->SetVertexComponents(ml->GetVertexComponents());
	refMesh->SetNumEdges(ml->GetNumEdges());

    
    int groupIndex;
    int numGroups = ml->GetNumGroups();
    for (groupIndex = 0; groupIndex < numGroups; groupIndex++)
    {
        nMeshGroup& group = refMesh->GetGroup(groupIndex);
        group = ml->GetGroupAt(groupIndex);
		group.SetFirstEdge(0);
		group.SetFirstIndex(0);
		group.SetFirstVertex(groupIndex *(vertexcount/ml->GetNumGroups()) );
		group.SetNumEdges(0);
		group.SetNumIndices(indexcount);
		group.SetNumVertices(vertexcount/ml->GetNumGroups());
    }

	refMesh->Load();

    float *verts = refMesh->LockVertices();
    n_assert(verts != NULL);
	/// @todo compute a bounding box 
	memcpy(verts,vertexbuffer,valuecount*sizeof(float));
    refMesh->UnlockVertices();


    // stuff the index data into the mesh
    ushort *indices = refMesh->LockIndices();
    n_assert(indices);
    memcpy(indices,indexbuffer,indexcount*sizeof(ushort));
    refMesh->UnlockIndices();

	ml->Close();

    delete [] vertexbuffer;
    delete [] indexbuffer;

    return true;
}

//------------------------------------------------------------------------------
/**
    Load the resources needed by this object.
*/
bool
nBlendMeshNode::LoadResources()
{
    if (nMaterialNode::LoadResources())
    {
        this->LoadMesh();
        return true;
    }
    return false;
}

//------------------------------------------------------------------------------
/**
    Unload the resources if refcount has reached zero.
*/
void
nBlendMeshNode::UnloadResources()
{
    nMaterialNode::UnloadResources();
    this->UnloadMesh();
}

//------------------------------------------------------------------------------
/**
    Indicate to scene server that we provide geometry
*/
bool
nBlendMeshNode::HasGeometry() const
{
    return true;
}

//------------------------------------------------------------------------------
/**
    Update geometry, set as current mesh in the gfx server and
    call nGfxServer2::DrawIndexed().

    - 15-Jan-04     floh    AreResourcesValid()/LoadResource() moved to scene server
*/
bool
nBlendMeshNode::RenderGeometry(nSceneServer* sceneServer, nRenderContext* renderContext)
{
    n_assert(sceneServer);
    n_assert(renderContext);
	if(refMesh->IsValid())
	{
		Interpolate(sceneServer,renderContext);
		return true;
	}
	return false;
}

//------------------------------------------------------------------------------
/**
    Set the resource name. The mesh resource name consists of the
    filename of the mesh.
*/
void
nBlendMeshNode::SetMesh(const char* name)
{
    n_assert(name);
    this->UnloadMesh();
    this->meshName = name;
}

//------------------------------------------------------------------------------
/**
*/
const char*
nBlendMeshNode::GetMesh() const
{
    return this->meshName.IsEmpty() ? 0 : this->meshName.Get();
}


//------------------------------------------------------------------------------
/**
    Set name of current animation

    @param     name of animation  
*/
void
nBlendMeshNode::SetAnimation(const char *str)
{
    n_assert(str);
	if(this->Anims == NULL)
	{
			return;
	}
	else
	{
		for(int i = 0; i < this->m_anims; i++)
		{
			if(strcmp(this->Anims[i].name, str)== 0)
			{
				// save old values
				this->m_pLastAnim = this->m_pCurrAnim;
				this->m_LastFrame = this->m_CurrFrame;
				// paste new values
				this->m_pCurrAnim = &(this->Anims[i]);
				//lastframe and currfarme need no handling

				break;
			}
		}

		this->m_AnimStart = -1.0f;
	}

	return;

}

//------------------------------------------------------------------------------
/**
    Set the resource loader used to load the mesh data.  If this is NULL, then
    the mesh is loaded through the default mesh loading code.
*/
void
nBlendMeshNode::SetMeshResourceLoader(const char* resourceLoaderPath)
{
    this->refMeshResourceLoader = resourceLoaderPath;
}

//------------------------------------------------------------------------------
/**
    Get the mesh resource loader.
    
    @return resource loader name or null when there is no resource loader
*/
const char *
nBlendMeshNode::GetMeshResourceLoader()
{
    if (this->refMeshResourceLoader.isvalid())
    {
        return this->refMeshResourceLoader.getname();
    }
    else
    {
        return 0;
    }
}

void nBlendMeshNode::Interpolate(nSceneServer* sceneServer, nRenderContext* renderContext)
{
	float dt;
    float min_t = 0.0f;
    float max_t = (this->m_pCurrAnim->frames/float(this->m_FPS));
	int frameA = 0;
	int frameB = 0;

	if(ipolLOD <= N_IPOL_IGNORE)
	{
		return;
	}
	else
	{
		if (max_t > 0.0) 
		{
			dt = this->kernelServer->GetTimeServer()->GetTime();
		
			//new animation set so start counting from here on (so t begins always at 0.0)
			if(this->m_AnimStart < 0.0f)
				this->m_AnimStart = dt + 0.0001f;

			dt = dt - this->m_AnimStart;//from here we did start counting so loop from this time on..

			// include animation speed modifier
			dt = dt * scale;

			// handle loop/oneshot behaviour
			if (repType == N_REPTYPE_LOOP) 
			{
				// in loop mode, normalize on loop time
				dt = dt - float(floor(dt/max_t)*max_t);
			}
			else // ONESHOOT
			{
				// clamp to vaild values
				if(dt > (max_t))
				{
					dt= max_t - 0.0001f;
				}
				else if(dt < 0.0f)
				{
					dt = 0.0001f;
				}	
			}

			// get the frames we are inbetween
			//int last = int(floor(t*this->m_FPS));
			int curr = int(ceil(dt*this->m_FPS));

			// adjustment if we came from a totaly other animation
			if(this->m_pLastAnim != this->m_pCurrAnim)
			{
				if(curr <= 1)
					m_CurrFrame = 1;
				else
				{
					this->m_pLastAnim = this->m_pCurrAnim;
					m_LastFrame = 1;
					m_CurrFrame = curr;
				};
			}
			else
			{
				// only needed if we are realy in another frame otherwise save 1 cycle
				if(m_CurrFrame != curr)
				{
					m_LastFrame = m_CurrFrame;
					m_CurrFrame = curr;
				}
			}
			// calculate frame positions in vertexbuffer
			nBlendAnimations *ptr = this->Anims;
			while(ptr != m_pLastAnim) 
			{
				frameA += ptr->frames;
				ptr++;
			}
			frameA += m_LastFrame -1;

			ptr = this->Anims;
			while(ptr != m_pCurrAnim)
			{
				frameB += ptr->frames;
				ptr++;
			}
			frameB += m_CurrFrame -1;
		}
		if(ipolLOD <= N_IPOL_FRAMES)
		{
			RenderFrame(sceneServer,renderContext,frameA);
		}
		else
		{
			// calculate lerp
			float l = dt - float((floor(dt/(m_invFPS)) / (this->m_FPS)));
			l = l/(m_invFPS);
			RenderLerp(sceneServer,renderContext,l, frameA,frameB);
		}
	}
}

void nBlendMeshNode::RenderFrame(nSceneServer* sceneServer, nRenderContext* renderContext, int frameA)
{
	if (!this->dynMesh.IsValid())
    {
        this->dynMesh.Initialize(nGfxServer2::TriangleList,
            refMesh->GetVertexComponents(), nMesh2::WriteOnly, false);
        n_assert(this->dynMesh.IsValid());
    }

	const nMeshGroup& curGroup = this->refMesh->GetGroup(frameA);
	float* dstVertices;
	float* currVertices;
    int    maxVertices;
    int curVertex = 0;
	int vCount = curGroup.GetNumVertices();

	// prepare
	this->dynMesh.Begin(dstVertices,maxVertices);
	currVertices = refMesh->LockVertices();
	currVertices = &currVertices[curGroup.GetFirstVertex()*refMesh->GetVertexWidth()]; 
	//render in one step
	if(vCount <= maxVertices)
	{
		memcpy(dstVertices,currVertices,vCount*refMesh->GetVertexWidth()*sizeof(float));
		curVertex = vCount;
	}
	else // render in multisteps
	{
		while(curVertex < vCount)
		{
			memcpy(dstVertices,currVertices,maxVertices*refMesh->GetVertexWidth()*sizeof(float));
			currVertices += maxVertices*refMesh->GetVertexWidth();
			dynMesh.Swap(maxVertices,dstVertices);
			curVertex +=maxVertices;
			// check for last update
			if((curVertex + maxVertices) > vCount )
			{
				curVertex = (vCount - curVertex);
				memcpy(dstVertices,currVertices,curVertex*refMesh->GetVertexWidth()*sizeof(float));
				break;// leave update loop;
			}
		}
	}
	// finish up
	refMesh->UnlockVertices();
	this->dynMesh.End(curVertex);
    //nGfxServer2* gfx = this->refGfxServer.get();
/* WRITE ONCE handling, not possible with read only mesh
    // @todo call geometry manipulators!
    n_assert(this->refMesh->IsValid());

    // render the mesh in normal mode (always at stream 0)
    refGfxServer->SetMesh(this->refMesh);

    // set the vertex and index range
    const nMeshGroup& curGroup = this->refMesh->GetGroup(frameA);
	const nMeshGroup& firstGrp = this->refMesh->GetGroup(0);
    refGfxServer->SetVertexRange(curGroup.GetFirstVertex(), curGroup.GetNumVertices());
    refGfxServer->SetIndexRange(firstGrp.GetFirstIndex(), firstGrp.GetNumIndices());
    refGfxServer->DrawIndexedNS(nGfxServer2::TriangleList);
	*/
}
void nBlendMeshNode::RenderLerp(nSceneServer* sceneServer, nRenderContext* renderContext, float lerp, int frameA, int frameB)
{
	if (!this->dynMesh.IsValid())
    {
        this->dynMesh.Initialize(nGfxServer2::TriangleList,
            refMesh->GetVertexComponents(), nMesh2::WriteOnly, false);
        n_assert(this->dynMesh.IsValid());
    }

	const nMeshGroup& AGroup = this->refMesh->GetGroup(frameA);
	const nMeshGroup& BGroup = this->refMesh->GetGroup(frameB);
	float* dstVertices;
	float* src0;
	float* src1;
	float* tmp;
    int    maxVertices;
    int curVertex = 0;
	int vCount = AGroup.GetNumVertices();

	// prepare
	this->dynMesh.Begin(dstVertices,maxVertices);
	tmp = refMesh->LockVertices();
	src0 = &tmp[AGroup.GetFirstVertex()*refMesh->GetVertexWidth()]; 
	src1 = &tmp[BGroup.GetFirstVertex()*refMesh->GetVertexWidth()]; 
	//render
    // the destination buffer must be written continously, because
    // it may be placed in uncached memory (AGP or video mem)
    // it may actually be better to keep coords, norms, etc 
    // in separate arrays -> DX8!
    for (int i=0; i<vCount; i++) 
    {
		if(refMesh->GetVertexComponents() & nMesh2::Coord)
		{
			dstVertices[0] = src0[0] + ((src1[0]-src0[0])*lerp);
			dstVertices[1] = src0[1] + ((src1[1]-src0[1])*lerp);
			dstVertices[2] = src0[2] + ((src1[2]-src0[2])*lerp);
			dstVertices += 3;
			src0 += 3;
			src1 += 3;
		}

		if(refMesh->GetVertexComponents() & nMesh2::Normal)
		{
			dstVertices[0] = src0[0] + ((src1[0]-src0[0])*lerp);
			dstVertices[1] = src0[1] + ((src1[1]-src0[1])*lerp);
			dstVertices[2] = src0[2] + ((src1[2]-src0[2])*lerp);
			dstVertices += 3;
			src0 += 3;
			src1 += 3;
		}

		if(refMesh->GetVertexComponents() & nMesh2::Texture)
		{
			dstVertices[0] = src0[0] + ((src1[0]-src0[0])*lerp);
			dstVertices[1] = src0[1] + ((src1[1]-src0[1])*lerp);
			dstVertices += 2;
			src0 += 2;
			src1 += 2;
		}

		if(refMesh->GetVertexComponents() & (nMesh2::Binormal|nMesh2::Tangent|nMesh2::JIndices|nMesh2::Color))
		{
			n_error("nBlendMeshNode invalid vertex component. Only supports vertex, texture and normal interpolation\n");
		}
		curVertex++;
		// swap if written enough
		if(curVertex >= maxVertices)
		{
			dynMesh.Swap(curVertex,dstVertices);
			curVertex = 0;
		}
    }
	// finish up
	refMesh->UnlockVertices();
	this->dynMesh.End(curVertex);
}
/*
void xy()
{

    // file name set?
    if (this->rsrc_path.GetPath())
    {
        // (re)-load mesh on demand
        if (!this->ref_vb.isvalid())
        {
            this->load_mesh();
        }
	}

	float tscale = this->scale;
    float min_t = 0.0f;
    float max_t = (this->m_pCurrAnim->frames/float(this->m_FPS));

    if (max_t > 0.0) 
	{
        // get current anim channel value
        nChannelContext* chnContext = sceneGraph->GetChannelContext();
        n_assert(chnContext);
        float t = chnContext->GetChannel1f(this->localChannelIndex);
	
		//new animation set so start counting from here on (so t begins always at 0.0)
		if(this->m_AnimStart < 0.0f)
			this->m_AnimStart = t + 0.0001f;

		t = t - this->m_AnimStart;//from here we did start counting so loop from this time on..

		// include animation speed modifier
		t = t * tscale;

        // handle loop/oneshot behaviour
        if (this->repType == N_REPTYPE_LOOP) 
        {
            // in loop mode, normalize on loop time
            t = t - float(floor(t/max_t)*max_t);
        }
		else // ONESHOOT
		{
			// clamp to vaild values
			if(t > (max_t))
			{
				t= max_t - 0.0001f;
			}
			else if(t < 0.0f)
			{
				t = 0.0001f;
			}	
		}

		// get the frames we are inbetween
		//int last = int(floor(t*this->m_FPS));
		int curr = int(ceil(t*this->m_FPS));

		// adjustment if we came from a totaly other animation
		if(this->m_pLastAnim != this->m_pCurrAnim)
		{
			if(curr <= 1)
				m_CurrFrame = 1;
			else
			{
				this->m_pLastAnim = this->m_pCurrAnim;
				m_LastFrame = 1;
				m_CurrFrame = curr;
			};
		}
		else
		{
			// only needed if we are realy in another frame otherwise save 1 cycle
			if(m_CurrFrame != curr)
			{
				m_LastFrame = m_CurrFrame;
				m_CurrFrame = curr;
			}
		}

		// calculate frame positions in vertexbuffer
		int frameA = 0;
		nMD2Animation *ptr = this->Anims;
		while(ptr != m_pLastAnim) 
		{
			frameA += ptr->frames;
			ptr++;
		}
		frameA += m_LastFrame -1;

		int frameB = 0;
		ptr = this->Anims;
		while(ptr != m_pCurrAnim)
		{
			frameB += ptr->frames;
			ptr++;
		}
		frameB += m_CurrFrame -1;

        // clear scene graph vertex buffer after child nodes wrote to it
        sceneGraph->SetVertexBuffer(0);

        // initialize dynamic vertex buffer if not happened yet
        if (!this->dyn_vb.IsValid()) 
        {
			n_assert(this->ref_vb.isvalid())
            this->dyn_vb.Initialize(this->ref_vb->GetVertexType(), this->m_vertices);
        
            // make sure that the vbuffers we get are bigger
            // then the source buffers, that's a compromise
            // we make to not make the code overly difficult
            if (this->m_vertices > this->dyn_vb.GetNumVertices()) 
            {
                n_printf("ERROR: source vertex buffers are greater then target vertex buffer!\n");
                n_error("Aborting.\n");
            }
        }

		// calculate lerp
		float l = t - float((floor(t/(1.0f/this->m_FPS)) / (this->m_FPS)));
		l = l/(1.0f/this->m_FPS);
   
        // generate and render interpolated vertex buffer
       this->interpolate(sceneGraph, l, frameA, frameB);

    }// end of if max_t > 0

    //sceneGraph->SetIndexBuffer(this->ref_ibuf.get());
    if (this->castShadows && this->refShadowCaster.isvalid())
    {
        sceneGraph->SetShadowCaster(this->refShadowCaster.get());
    }
}
*/