//------------------------------------------------------------------------------
//  (C)	2002-2004	tom@3d-inferno.com
//------------------------------------------------------------------------------

#include "kernel/ntimeserver.h"
#include "water/nwaterserver.h"


nNebulaScriptClass(nWaterServer, "nroot");

//------------------------------------------------------------------------------
/**
    Initializes gfxServer reference, the fps update setting and all
	variables to zero or NULL as needed
*/
nWaterServer::nWaterServer()
: m_A(0),
m_B(0),
m_C(0),
ref_ibuf(NULL),
ref_i3to1buf(NULL),
m_Choppy(NULL),
m_Gerstner(NULL),
m_Updated(0)
{
//	this->refGfx     = "/sys/servers/gfx";
	this->SetFPS(6);
	m_last = m_delta = 0.0f;
}

//------------------------------------------------------------------------------
/**
    Release by waterserver allocated mem. (water instances and index buffers)
*/
nWaterServer::~nWaterServer()
{
	// normally all childs should have removed their references first
	if(m_Gerstner || m_Choppy)
		n_printf("~nWaterserver: Not alll references were released\n");

	if(m_Choppy)
	{
		n_delete (m_Choppy);
		m_Choppy = NULL;
	}
	if(m_Gerstner)
	{
		n_delete (m_Gerstner);
		m_Gerstner = NULL;
	}
	if(ref_ibuf)
		n_delete_array(ref_ibuf);
	if(ref_i3to1buf)
		n_delete_array(ref_i3to1buf);
};

//------------------------------------------------------------------------------
/**
	Return reference to @see nComplex array of @see SampleSize which describes the deviation
	in Y (up and down) for a corresponding water point.
	@todo move this function to private section. Use .real() to get world deviation of complex.

    @return                 pointer to nComplex array of Y deviation
*/
const nComplex *nWaterServer::getCMap(){return m_Choppy->getMap();};
//------------------------------------------------------------------------------
/**
	Return reference to @see nComplex array of @see SampleSize which describes the deviation
	in X (left and right) for a corresponding water point.
	@todo move this function to private section. Use .real() to get world deviation of complex.

    @return                 pointer to nComplex array of X deviations
*/
const nComplex *nWaterServer::getXMap(){return m_Choppy->getDeltaMapX();};
//------------------------------------------------------------------------------
/**
	Return reference to @see nComplex array of @see SampleSize which describes the deviation
	in Z (fro and aft) for a corresponding water point.
	@todo move this function to private section. Use .real() to get world deviation of complex.

    @return                 pointer to nComplex array of Z deviations
*/
const nComplex *nWaterServer::getZMap(){return m_Choppy->getDeltaMapZ();};
//------------------------------------------------------------------------------
/**
	Return reference to @see vector3 array of @see SampleSize which describes the position
	for a corresponding water point.
	@todo move this function to private section. Use .real() to get world deviation of complex.

    @return                 pointer to nComplex array of X deviations
*/
vector3 *nWaterServer::getGMap(){return m_Gerstner->getVertices();};
//------------------------------------------------------------------------------
/**
    Return the number of total @see nWaternode references to this waterserver

    @return                 count of all algorithm references summed up
*/
int nWaterServer::GetRefs(){return m_A + m_B + m_C;};
//------------------------------------------------------------------------------
/**
    Return the choppyness factor of the ChoppyWaves water algorithm.

    @return                 choppyness value
*/
float nWaterServer::GetChopp(){ return (m_A||m_B) ? m_Choppy->getLambda() : 0.0f;	};
/**
    Set the abolute choppyness factor of the choppy waves algorithm for the water
	server. The choppyness dictates how big the deviation in X or Z will be.
	-0.15 is quite good smaller values lead to more choppyness, higher values to less.

    @param  _t      factor of choppyness, -0.15 is quite good smaller values lead to more choppyness, higher values to less

*/
void nWaterServer::SetChopp(float _t){ if(m_A||m_B) m_Choppy->setLambda(_t);};
//------------------------------------------------------------------------------
/**
    Toggels usage of 3to1 mesh bridge. 

    @param  flag      true: use 3to1 mesh, false do not use 3to1 mesh
*/
void nWaterServer::Create3to1Mesh(bool flag){m_3to1Mesh =flag;};
//------------------------------------------------------------------------------
/**
    Register an interrest for upate of provided water algorithm. Must call RemWaterRef
	to release the water algorithm interrest. @see RemWaterRef.
	Creates water algorithm instances if a water algorithm is first referenced.

    @param  _ref      Algorithm reference (0: StatisticalWaves, 1:ChoppyWaves, 2:Gerstnerwaves, 3:3to1 ChoppyWaves
    @return           true if success, false otherwise
*/
bool nWaterServer::AddWaterRef(int _ref)
{
	if(!this->ref_ibuf)
		initIndex();
	
	switch(_ref)
	{
	case N_STATISTIC:
n_printf("statistic added %d\n", m_A);
		if((!m_Choppy) && (m_A ==0) )
		{
			m_Choppy = n_new(ChoppyWaves(-0.15f, SampleSize, SampleSize,
				0.0008f, vector2(30,10), 100.0f, 100.0f));
			updateStatistic();
		}
//		if(!m_A)
//			updateStatistic();
		m_A++; 
		return true;
	case N_CHOPPY:
n_printf("choppy added %d\n", m_B);
		if((!m_Choppy) && m_B ==0 )
		{ 
			m_Choppy = n_new(ChoppyWaves(-0.15f, SampleSize, SampleSize, 
				0.0008f, vector2(30,10), 100.0f, 100.0f));
			updateChoppy();
		}
//		if(!m_B)
//			updateChoppy();
		m_B++; return true;
	case N_GERSTNER:
n_printf("gerstner added %d\n", m_C);
		if((!m_Gerstner) && m_C ==0 )
		{
			m_Gerstner = n_new(GerstnerWaves(SampleSize, SampleSize));
			m_Gerstner->addWave(vector2(1, 1), 5, 0.1f, 1);
			m_Gerstner->addWave(vector2(2, 1), 20, 0.8f, 3);
			m_Gerstner->addWave(vector2(3, 2), 10, 0.3f, 2);
			m_Gerstner->addWave(vector2(2, 1), 40, 1.4f, 9);
			updateGerstner();
		}
//		if(!m_C)
//			updateGerstner();
		m_C++; return true;
//	case N_CHOPPY_3TO1:
//		m_3to1Mesh=true;return true;
	default: return false;
	}
}

//------------------------------------------------------------------------------
/**
    Release/remove an interrest for upate of provided water algorithm. Must have been 
	called by @see AddWaterRef to register the water algorithm interrest first. 
	Releases water algorithm instances if water algorithm is not referenced any longer.

    @param  _ref      Algorithm reference (0: StatisticalWaves, 1:ChoppyWaves, 2:Gerstnerwaves, 3:3to1 ChoppyWaves
    @return           true if success, false otherwise
*/
bool nWaterServer::RemWaterRef(int _ref)
{
	switch(_ref)
	{
	case N_STATISTIC:
n_printf("statistic removed %d\n",m_A);
		m_A--;
		if(!m_A)
		{
			if(!m_B)
			{
				n_delete(m_Choppy);
				m_Choppy = NULL;
			}
		}
		n_assert(m_A>-1);return true;
	case N_CHOPPY:
n_printf("choppy removed %d\n", m_B);
		m_B--;
		if(!m_A)
		{
			if(!m_B)
			{
				n_delete(m_Choppy);
				m_Choppy = NULL;
			}
		}
		n_assert(m_B>-1);return true;
	case N_GERSTNER:
n_printf("gerstner removed %d\n", m_C);
		m_C--;
		if(!m_C)
		{
			n_delete(m_Gerstner);
			m_Gerstner = NULL;
		}
		n_assert(m_C>-1);return true;
	default: return false;
	}
}

//------------------------------------------------------------------------------
/**
	Returns the update frequency of water algorithms, stating how any frames per second
	will be used for update calculations of the different water algorithms.

    @return           Integer value of update frames per second
*/
int nWaterServer::GetFPS()
{
	return this->m_FPS;
}

//------------------------------------------------------------------------------
/**
	Sets the update frequency of water algorithms, stating how any frames per second
	will be used for update calculations of the different water algorithms. Precomutes
	the m_delta member as 1/FPS for avoiding float divisions.

	@param		i	  Integer value of  update frames per second
*/
void nWaterServer::SetFPS(int i)
{
	n_assert(i > 0)
    this->m_FPS = i;
	this->m_delta = (float)1/i;
}


//------------------------------------------------------------------------------
/**
	Central update function. Actuall time is compared against last update. If the
	actual time exceeds the update intervals set by @see SetFPS then a new update
	cycle begins. In the first frame of an update cycle only the choppy waves are
	actuallized with the statistical mesh, in the next frame the choppywaves mesh
	gets updated, in the fourth frame the gerstner waves and its corresponding mesh
	are updated (this could decoupled too). Finally the update cycle is finalyzed by
	adjusting the last update time tothe current frame time.
	Thus we split the workload to several frames in hope for a minor lag between one
	or two frames only.

	@todo Assert frame interval is really used, not sure if it is at the moment

	@return			  Returns true if update occured
*/
bool nWaterServer::Update(void)
{
	float _t = this->kernelServer->GetTimeServer()->GetTime();
	if((_t - m_last) > m_delta)
	{
		//update each working algorithm in a different time frame for balance
		switch(m_Updated)
		{
		case 0:
			if(m_Choppy)
				m_Choppy->update(_t);
			if(m_A)
				this->updateStatistic();
//			if(this->m_3to1Mesh)//test only
//				this->update3to1(1);
			m_Updated++;
			break;
		case 1:
			if(m_B)
			{
				this->updateChoppy();
//				if(this->m_3to1Mesh)
//				this->update3to1(1);
			}
				m_Updated++;
			break;
		case 2:
			if(m_C && m_Gerstner)
			{
				m_Gerstner->update(_t);
				this->updateGerstner();
			}
		default:
			m_last = _t;
			m_Updated=0;
		}
		return true;
	}
	else
		return false;
}

//------------------------------------------------------------------------------
/**
	Updates the choppywaves mesh buffer referenced by all nWaterNodes using the
	choppy waves algorithm. Prior to this, the index buffer short buffer must 
	have been filled and the choppywaves instance at least once updated. Each
	time this function is called the mesh is filled with the new updated vertex
	data.
	
	@todo the formula to compute vertex normals is fake but looks good enough, this should be replaced sometime with a real normal.
*/
void nWaterServer::updateChoppy()
{
	vector3 v1,vbase,no;
	const nComplex * xmap = getXMap();
	const nComplex * ymap = getCMap();
	const nComplex * zmap = getZMap();

	//check if buffer is valid
	if(!this->c_vb.isvalid())
	{
		char iname[256];
		//that's how waternodes will find the buffer so don't change the name
		//if you don't change it in nwaternode.cc!
		sprintf(iname,"nWaterNode_%d_%d",N_CHOPPY,SampleSize);
		//create will also find it if it exists, this is different from n1
		this->c_vb = nGfxServer2::Instance()->NewMesh(iname);
		n_assert(c_vb);
		this->c_vb->SetUsage(this->GetMeshUsage());
		this->c_vb->SetVertexComponents(VertType);
		//here the index and vertex counts are filled
		this->initGroup(this->c_vb);
		//creates empty buffers as the filename of the mesh was left out
		this->c_vb->Load();
		//fill in the index buffer of new mesh instance
		this->initChoppyIndex();
	}
	// update verte buffer part

	//lock it for overwriting
	float * ptr = this->c_vb->LockVertices();
	//ulong ccol = n_f2rgba(1.0f,1.0f,1.0f,1.0f); 

	int tot=0;//total vertices processed
	//remember this stretches a 32 *32 complex field to a 33*33 vertex field
	for(int rows=0, a, b; rows <= SampleSize; rows++)
	{//a and b are used to have tilable sides (same verts at oposite side)
		if(rows>=SampleSize)
			b=0;
		else
			b=rows;
		for(int cols=0; cols <=SampleSize; cols++,tot++)
		{	
			if(cols>=SampleSize)
				a=0;
			else
				a=cols;
		
			if(a==0)//no x deviation
			{
				v1.x=(float)cols;
			}
			else
			{
				v1.x=(float) a + xmap[a + (b * SampleSize)].real()*0.4f;
			}
			
			if(b==0)//||rows==(SampleSize))//no z deviation
			{
				v1.z=(float)rows;
			}
			else //z deviation
			{
				v1.z=(float) b + zmap[a + (b * SampleSize)].real()*0.4f;
			}
			v1.y=ymap[a + (b * SampleSize)].real()*0.04f;

			ptr[0] = v1.x;
			ptr[1] = v1.y;
			ptr[2] = v1.z;

			ptr[3] = 0.0f;
			ptr[4] = v1.y*v1.y;//fake normals but fast
			ptr[5] = 0.0f;

			//color = &ccol;// colors are all the same this could be chnaged in future

			ptr[6] = ((float)(cols)/(SampleSize+1));
			ptr[7] = ((float)(rows)/(SampleSize+1));

			ptr += 8;
		}//end of cols
	}//end of rows
	//relesae lock
	this->c_vb->UnlockVertices();
}

//------------------------------------------------------------------------------
/**
	Updates the statisticwaves mesh buffer referenced by all nWaterNodes using the
	statistic waves algorithm. Prior to this, the index buffer short buffer must 
	have been filled and the statisticwaves instance at least once updated. Each
	time this function is called the mesh is filled with the new updated vertex
	data.
	
	@todo the formula to compute vertex normals is fake but looks good enough, this should be replaced sometime with a real normal.
*/
void nWaterServer::updateStatistic()
{
	float height;
	const nComplex * map = getCMap();
	
	if(!this->s_vb.isvalid())
	{
		char iname[256];
		sprintf(iname,"nWaterNode_%d_%d",N_STATISTIC,SampleSize);
		//first try to fid it then create a new !
		this->s_vb = nGfxServer2::Instance()->NewMesh(iname);
		n_assert(s_vb);
		this->s_vb->SetUsage(this->GetMeshUsage());
		this->s_vb->SetVertexComponents(VertType);
		this->initGroup(this->s_vb);
		this->s_vb->Load();
		this->initStatisticalIndex();
	}
		
	float* ptr = this->s_vb->LockVertices();
//	ulong ccol = n_f2rgba(1.0f,1.0f,1.0f,1.0f); 

	int tot=0;//total vertices processed
	//remember this stretches a 32 *32 complex field to a 33*33 vertex field
	for(int rows=0, a, b; rows <= SampleSize; rows++)
	{//a and b are used to have tilable sides (same verts at oposite side)
		if(rows>=SampleSize)
			b=0;
		else
			b=rows;
		for(int cols=0; cols <=SampleSize; cols++,tot++)
		{	
			if(cols>=SampleSize)
				a=0;
			else
				a=cols;

			height=map[a + (b * SampleSize)].real()*0.04f;
			ptr[0] = (float)cols;
			ptr[1] = height;
			ptr[2] = (float)rows;
			//normal base is always 0 in x and z direction, not normalized like choppy
			ptr[3] = 0.0f;
			ptr[4] = height*height;//for always positive values
			ptr[5] = 0.0f;

			//color = &ccol;// colors are all the same this could be chnaged in future

			ptr[6] = ((float)(cols)/(SampleSize+1));
			ptr[7] = ((float)(rows)/(SampleSize+1));

			ptr += 8;
		}//end of cols
	}//end of rows
	this->s_vb->UnlockVertices();
}


//------------------------------------------------------------------------------
/**
	Updates the gerstnerwaves mesh buffer referenced by all nWaterNodes using the
	gerstnerwaves algorithm. Prior to this, the index buffer short buffer must 
	have been filled and the gerstnerwaves instance at least once updated. Each
	time this function is called the mesh is filled with the new updated vertex
	data.
	
	@todo the formula to compute vertex normals is fake but looks good enough, this should be replaced sometime with a real normal.
*/
void nWaterServer::updateGerstner()
{
	vector3 v1,vbase,no;
	const vector3 * map = getGMap();
 
   	if(!this->g_vb.isvalid())
	{
		char iname[256];
		sprintf(iname,"nWaterNode_%d_%d",N_GERSTNER,SampleSize);
		//first try to find it then create a new !
		this->g_vb = nGfxServer2::Instance()->NewMesh(iname);
		n_assert(g_vb);
		this->g_vb->SetUsage(this->GetMeshUsage());
		this->g_vb->SetVertexComponents(VertType);
		this->initGroup(this->g_vb);
		this->g_vb->Load();
		this->initGerstnerIndex();
	}
		
	float * ptr = this->g_vb->LockVertices();
	//ulong ccol = n_f2rgba(1.0f,1.0f,1.0f,1.0f); 

	int tot=0;//total vertices processed
	//remember this stretches a 32 *32 complex field to a 33*33 vertex field
	for(int rows=0, a, b; rows <= SampleSize; rows++)
	{//a and b are used to have tilable sides (same verts at oposite side)
		if(rows>=SampleSize)
			b=0;
		else
			b=rows;
		for(int cols=0; cols <=SampleSize; cols++,tot++)
		{	
			if(cols>=SampleSize)
				a=0;
			else
				a=cols;
			
			if(a==0)//no x deviation
			{
				v1.x=(float)cols;
			}
			else
			{
				v1.x=map[a + (b * SampleSize)].x;//(float) a + map[a + (b * SampleSize)].x*0.02f;
//			n_clamp(v1.x, 0.1f,31.9f);
			}
			
			
			if(b==0)//||rows==(SampleSize))//no z deviation
			{
				v1.z=(float)rows;
			}
			else //z deviation
			{
				v1.z=map[a + (b * SampleSize)].z;//(float) b + map[a + (b * SampleSize)].z*0.02f;
//				n_clamp(v1.x, 0.1f,31.9f);
			}
			v1.y=map[a + (b * SampleSize)].y;

			ptr[0] = v1.x;
			ptr[1] = v1.y;
			ptr[2] = v1.z;

			ptr[3] = 0.0f;//v1.x - (float)cols;
			ptr[4] = v1.y*v1.y;
			ptr[5] = 0.0f;//v1.z - (float)rows;

			//color = &ccol;// colors are all the same this could be chnaged in future

			ptr[6] = ((float)(cols)/(SampleSize+1));
			ptr[7] = ((float)(rows)/(SampleSize+1));

			ptr+=8;
		}//end of cols
	}//end of rows
	this->g_vb->UnlockVertices();
}


//------------------------------------------------------------------------------
/**
	Fills the unsigned short index buffer with index values for a SampleSize times
	SampleSize Grid. 
*/
void nWaterServer::initIndex(void)
{
    n_assert(!this->ref_ibuf);
	
	int i,j;
	int act_v = ((SampleSize+1) * (SampleSize));// ((SampleSize+1) * (SampleSize));//act_v = (actSize)*(actSize - 1);
	int ass = ((SampleSize)*(SampleSize) * 6);
	// create index buffer
	this->ref_ibuf = n_new(ushort[((SampleSize)*(SampleSize) * 6)]);
	for (j=0, i=0 ; i < act_v; i++) 
	{	
		if(((i+1)%(SampleSize+1))==0)
		{
			continue;
		}
		this->ref_ibuf[j] = i;
		this->ref_ibuf[j+1] = i+(SampleSize+1);
		this->ref_ibuf[j+2] = i+1;
		
		j+=3;
		this->ref_ibuf[j] = i+1;
		this->ref_ibuf[j+1] = i+(SampleSize+1);
		this->ref_ibuf[j+2] = i+(SampleSize+1)+1;
		
		j+=3;
		n_assert(j <= ass);
	}
}

//------------------------------------------------------------------------------
/**
	Initializes theee passed nMesh2 instance with index and vertex counts.
	The water uses only one group in each mesh and no edge information.
*/
void nWaterServer::initGroup(nMesh2 * pmesh)
{
	pmesh->SetVertexComponents(VertType);
	pmesh->SetNumGroups(1);
	pmesh->SetNumEdges(0);
	pmesh->SetNumIndices((SampleSize)*(SampleSize) * 6);
	pmesh->SetNumVertices((SampleSize	+1)*(SampleSize	+1));

	nMeshGroup& grp = pmesh->GetGroup(0);
	grp.SetFirstEdge(0);
	grp.SetNumEdges(0);
	grp.SetFirstIndex(0);
	grp.SetNumIndices((SampleSize)*(SampleSize) * 6);
	grp.SetFirstVertex(0);
	grp.SetNumVertices((SampleSize	+1)*(SampleSize	+1));
}

//------------------------------------------------------------------------------
/**
	Fills the index information of the statistical mesh instance.
	We only have one base index but the usage of the new meshes is each mesh must 
	use his own index :-(
*/
void nWaterServer::initStatisticalIndex()
{
	int act_v = ((SampleSize)*(SampleSize) * 6);// ((SampleSize+1) * (SampleSize));//act_v = (actSize)*(actSize - 1);
	n_assert(this->ref_ibuf);

	ushort 	* ptr = this->s_vb->LockIndices();
	memcpy(ptr,this->ref_ibuf,(act_v*sizeof(ushort)));
	this->s_vb->UnlockIndices();
}

//------------------------------------------------------------------------------
/**
	Fills the index information of the choppy mesh instance.
	We only have one base index but the usage of the new meshes is each mesh must 
	use his own index :-(
*/
void nWaterServer::initChoppyIndex()
{
	int act_v = ((SampleSize)*(SampleSize) * 6);// ((SampleSize+1) * (SampleSize));//act_v = (actSize)*(actSize - 1);
	n_assert(this->ref_ibuf);

	ushort 	* ptr = this->c_vb->LockIndices();
	memcpy(ptr,this->ref_ibuf,(act_v*sizeof(ushort)));
	this->c_vb->UnlockIndices();
}

//------------------------------------------------------------------------------
/**
	Fills the index information of the gerstner mesh instance.
	We only have one base index but the usage of the new meshes is each mesh must 
	use his own index :-(
*/
void nWaterServer::initGerstnerIndex()
{
	int act_v = ((SampleSize)*(SampleSize) * 6);// ((SampleSize+1) * (SampleSize));//act_v = (actSize)*(actSize - 1);
	n_assert(this->ref_ibuf);

	ushort 	* ptr = this->g_vb->LockIndices();
	memcpy(ptr,this->ref_ibuf,(act_v*sizeof(ushort)));
	this->g_vb->UnlockIndices();
}

//------------------------------------------------------------------------------
/**
	Initializes unsigned short array for bridge mesh between 3*3 water meshes to a
	mesh scaled to 3 times its size.
	Not used at the moment.

	@todo test and validate 3to1 still is ok. Ev.could be used as bridge to a static mesh
*/
void nWaterServer::init3to1Index(void)
{
    n_assert(!this->ref_i3to1buf);
	this->ref_i3to1buf = n_new(ushort[SampleSize*4*12]);

	for (int a=0, b=(SampleSize*12),i=0; i<(SampleSize*4*12); i+=24,a+=6,b+=2) 
	{	
		this->ref_i3to1buf[i] = b;	
		this->ref_i3to1buf[i+1] = a; 
		this->ref_i3to1buf[i+2] = a+1;
//n_printf("%d %d %d\n",b,a,a+1);
		this->ref_i3to1buf[i+3] = a+1; 
		this->ref_i3to1buf[i+4] = a+2; 
		this->ref_i3to1buf[i+5] = b;
//n_printf("%d %d %d\n",a+1,a+2,b);
		this->ref_i3to1buf[i+6] = b; 
		this->ref_i3to1buf[i+7] = a+2; 
		this->ref_i3to1buf[i+8] = b+1;
//n_printf("%d %d %d\n",b,a+2,b+1);
		this->ref_i3to1buf[i+9] = b+1; 
		this->ref_i3to1buf[i+10] = a+2; 
		this->ref_i3to1buf[i+11] = a+3;
//n_printf("%d %d %d\n",b+1,a+2,a+3);
		this->ref_i3to1buf[i+12] = a+3; 
		this->ref_i3to1buf[i+13] = a+4; 
		this->ref_i3to1buf[i+14] = b+1;
//n_printf("%d %d %d\n",a+3,a+4,b+1);
		this->ref_i3to1buf[i+15] = b+1; 
		this->ref_i3to1buf[i+16] = a+4; 
		if(i>=(SampleSize*48)-24)
		{	
			this->ref_i3to1buf[i+17] = ((SampleSize)*12);
//n_printf("%d %d %d\n",b+1,a+4,((SampleSize)*12));
			this->ref_i3to1buf[i+18] = ((SampleSize)*12);
			this->ref_i3to1buf[i+19] = a+4; 
			this->ref_i3to1buf[i+20] = a+5;
//n_printf("%d %d %d\n",((SampleSize)*12),a+4,a+5);
			this->ref_i3to1buf[i+21] = a+5; 
			this->ref_i3to1buf[i+22] = a+6;
			this->ref_i3to1buf[i+23] = 0;
//n_printf("%d %d %d     i:%d\n",a+5,a+6,0,i+23);
		}
		else
		{
			this->ref_i3to1buf[i+17] = b+2;
//n_printf("%d %d %d\n",b+1,a+4,b+2);
			this->ref_i3to1buf[i+18] = b+2;
			this->ref_i3to1buf[i+19] = a+4; 
			this->ref_i3to1buf[i+20] = a+5;
//n_printf("%d %d %d\n",b+2,a+4,a+5);
			this->ref_i3to1buf[i+21] = a+5; 
			this->ref_i3to1buf[i+22] = a+6;
			this->ref_i3to1buf[i+23] = b+2;
//n_printf("%d %d %d     i:%d\n",a+5,a+6,b+2,i+23);
		}
	}
}

//------------------------------------------------------------------------------
/**
	Updates mesh used for bridging 3*3 to mesh scaled to three times the size.
	can be used to close the gap of 3 to 1 meshes.
	xxxxxxxxxx-------+
	x  |  |  x       |
	x--+--+--x       |
	x  |  |  x       |
	x--+--+--x       |
	x  |  |  x       |
	xxxxxxxxxx-------+
	
	@todo test functionality and assert correct work, ev. change to bridge to a static mesh

	@param		type	type of water algorithm the bridge is intended for, NOT USED
*/
void nWaterServer::update3to1(int type)
{
	if(!this->ref_i3to1buf)
		init3to1Index();// just only one needed, all three use same index

   	if(!this->c_3to1vb.isvalid())
	{
		char iname[256];
		sprintf(iname,"nWaterNode_%d_%d",N_CHOPPY_3TO1, SampleSize);
		//first try to find it then create a new !
		this->c_3to1vb = nGfxServer2::Instance()->NewMesh(iname);
		this->c_3to1vb = nGfxServer2::Instance()->NewMesh(iname);
		n_assert(c_3to1vb);
		this->c_3to1vb->SetUsage(this->GetMeshUsage());
		this->c_3to1vb->SetVertexComponents(VertType);
	//	this->initGroup(this->c_3to1vb);
	// FIXME special group !
		this->c_3to1vb->Load();
		this->init3to1Index();
	}

	const nComplex * xmap = getXMap();
	const nComplex * ymap = getCMap();
	const nComplex * zmap = getZMap();

	float * ptr = this->c_3to1vb->LockVertices();
	//ulong ccol = n_f2rgba(1.0f,1.0f,1.0f,1.0f);
	
	int x,z,i,g;
	g=-1;
//	static bool walk=true;
	// walk right
	for(i=0,x=0,z=0;i<(SampleSize *3);i++,x++)
	{
		g++;
		ptr[0]=(float)x + xmap[x%SampleSize].real() * 0.04f;
		ptr[1]=ymap[x%SampleSize].real() * 0.04f;
		ptr[2]=(float)z;
//if(walk)
//n_printf("g: %d\tx: %f\t z:%f\n",g,((float)x + xmap[x%SampleSize].real() * 0.04f),(float)z);
		ptr[3] = 0.0f;
		ptr[4] = ymap[x%SampleSize].real() * ymap[x%SampleSize].real() * 0.0016f;
		ptr[5] = 0.0f;

		//color = &ccol;// colors are all the same this could be chnaged in future

		ptr[6] = (float)(x%SampleSize)/SampleSize;
//if(walk)
//n_printf("t: u:%f\tv:%f\n",(float)(x%SampleSize)/SampleSize,0.0f);
		ptr[7] = 0.0f;

		ptr += 8;
	}
	//walk down
	for(i=0;i<(SampleSize *3);i++,z++)
	{
		g++;
		ptr[0]=(float)x;
		ptr[1]=ymap[(z%SampleSize)*SampleSize].real() * 0.04f;
		ptr[2]=(float)z + zmap[(z%SampleSize)*SampleSize].real() * 0.04f;
//if(walk)
//n_printf("g: %d\tx: %f\t z:%f\n",g,(float)x,(float)z + zmap[(z%SampleSize)*SampleSize].real() * 0.04f);		
		ptr[3] = 0.0f;
		ptr[4] = ymap[(z%SampleSize)*SampleSize].real() * ymap[(z%SampleSize)*SampleSize].real() * 0.0016f;
		ptr[5] = 0.0f;

		//color = &ccol;// colors are all the same this could be chnaged in future

		ptr[6] = 1.0f;
		ptr[7] = (float)(z%SampleSize)/SampleSize;
//if(walk)
//n_printf("t: u:%f\tv:%f\n",1.0f, (float)(z%SampleSize)/SampleSize);

		ptr += 8;
	}
	//walks left on bottom
	for(i=0;i<(SampleSize *3);i++,x--)
	{
		g++;
		ptr[0]=(float)x + xmap[x%SampleSize].real() * 0.04f;
		ptr[1]=ymap[x%SampleSize].real() * 0.04f;
		ptr[2]=(float)z;
//if(walk)
//n_printf("g: %d\tx: %f\t z:%f\n",g,(float)x + xmap[x%SampleSize].real() * 0.04f,(float)z);		
		ptr[3] = 0.0f;
		ptr[4] = ymap[x%SampleSize].real() * ymap[x%SampleSize].real() * 0.0016f;
		ptr[5] = 0.0f;

		//color = &ccol;// colors are all the same this could be chnaged in future

		ptr[6] = (float)(x%SampleSize)/SampleSize;
		ptr[7] = 1.0f;
//if(walk)
//n_printf("t: u:%f\tv:%f\n",(float)(x%SampleSize)/SampleSize,0.0f);

		ptr += 8;
	}
	//walk up to start
	for(i=0;i<(SampleSize *3);i++,z--)
	{
		g++;
		ptr[0]=(float)x;
		ptr[1]=ymap[(z%SampleSize)*SampleSize].real() * 0.04f;
		ptr[2]=(float)z + zmap[(z%SampleSize)*SampleSize].real() * 0.04f;
//if(walk)
//n_printf("g: %d\tx: %f\t z:%f\n",g,(float)x,(float)z + zmap[(z%SampleSize)*SampleSize].real() * 0.04f);		
		ptr[3] = 0.0f;
		ptr[4] = ymap[(z%SampleSize)*SampleSize].real() * ymap[(z%SampleSize)*SampleSize].real() * 0.0016f;
		ptr[5] = 0.0f;

		//color = &ccol;// colors are all the same this could be chnaged in future

		ptr[6] = 0.0f;
		ptr[7] = (float)(z%SampleSize)/SampleSize;
//if(walk)
//n_printf("t: u:%f\tv:%f\n",0.0f,(float)(z%SampleSize)/SampleSize);

		ptr += 8;
	}
	//now the 4 sides of the outside mesh
	for(i=0,x=0,z=0;i<SampleSize;i++,x++)
	{
		g++;
		ptr[0]=((float)x + xmap[x%SampleSize].real() * 0.04f)*3.0f;
		ptr[1]=ymap[x%SampleSize].real() * 0.04f;
		ptr[2]=(float)z;
//if(walk)
//n_printf("g: %d\tx: %f\t z:%f\n",g,((float)x + xmap[x%SampleSize].real() * 0.04f)*3.0f,(float)z);		
		ptr[3] = 0.0f;
		ptr[4] = ymap[x%SampleSize].real() * ymap[x%SampleSize].real() * 0.0016f;
		ptr[5] = 0.0f;

		//color = &ccol;// colors are all the same this could be chnaged in future
		//UV must be scaled by shadernode
		ptr[6] = (float)(x%SampleSize)/SampleSize;
		ptr[7] = 1.0f;
//if(walk)
//n_printf("t: u:%f\tv:%f\n",(float)(x%SampleSize)/SampleSize,1.0f);
	
		ptr += 8;
	}
	//walk down
	for(i=0;i<SampleSize;i++,z++)
	{
		g++;
		ptr[0]=(float)x*3.0f;
		ptr[1]=ymap[(z%SampleSize)*SampleSize].real() * 0.04f;
		ptr[2]=((float)z + zmap[(z%SampleSize)*SampleSize].real() * 0.04f)*3.0f;
//if(walk)
//n_printf("g: %d\tx: %f\t z:%f\n",g,(float)x*3.0f,((float)z + zmap[(z%SampleSize)*SampleSize].real() * 0.04f)*3.0f);		
		ptr[3] = 0.0f;
		ptr[4] = ymap[(z%SampleSize)*SampleSize].real() * ymap[(z%SampleSize)*SampleSize].real() * 0.0016f;
		ptr[5] = 0.0f;

		//color = &ccol;// colors are all the same this could be chnaged in future

		ptr[6] = 0.0f;
		ptr[7] = (float)(z%SampleSize)/SampleSize;
//if(walk)
//n_printf("t: u:%f\tv:%f\n",0.0f,(float)(z%SampleSize)/SampleSize);
	
		ptr += 8;
	}
	//walks left on bottom
	for(i=0;i<SampleSize;i++,x--)
	{
		g++;
		ptr[0]=((float)x + xmap[x%SampleSize].real() * 0.04f)*3.0f;
		ptr[1]=ymap[x%SampleSize].real() * 0.04f;
		ptr[2]=(float)z*3.0f;
//if(walk)
//n_printf("g: %d\tx: %f\t z:%f\n",g,((float)x + xmap[x%SampleSize].real() * 0.04f)*3.0f,(float)z*3.0f);		
		ptr[3] = 0.0f;
		ptr[4] = ymap[x%SampleSize].real() * ymap[x%SampleSize].real() * 0.0016f;
		ptr[5] = 0.0f;

		//color = &ccol;// colors are all the same this could be chnaged in future

		ptr[6] = (float)(x%SampleSize)/SampleSize;
		ptr[7] = 0.0f;
//if(walk)
//n_printf("t: u:%f\tv:%f\n",(float)(x%SampleSize)/SampleSize,0.0f);
	
		ptr += 8;
	}
	//walk up to start
	for(i=0;i<SampleSize;i++,z--)
	{
		g++;
		ptr[0]=(float)x*3.0f;
		ptr[1]=ymap[(z%SampleSize)*SampleSize].real() * 0.04f;
		ptr[2]=((float)z + zmap[(z%SampleSize)*SampleSize].real() * 0.04f)*3.0f;
//if(walk)
//n_printf("g: %d\tx: %f\t z:%f\n",g,(float)x*3.0f,((float)z + zmap[(z%SampleSize)*SampleSize].real() * 0.04f)*3.0f);		
		ptr[3] = 0.0f;
		ptr[4] = ymap[(z%SampleSize)*SampleSize].real() * ymap[(z%SampleSize)*SampleSize].real() * 0.0016f;
		ptr[5] = 0.0f;

		//color = &ccol;// colors are all the same this could be chnaged in future

		ptr[6] = 1.0f;
		ptr[7] = (float)(z%SampleSize)/SampleSize;
//if(walk)
//n_printf("t: u:%f\tv:%f\n",1.0f,(float)(z%SampleSize)/SampleSize);

		ptr += 8;
	}
//walk=false;
	this->c_3to1vb->UnlockVertices();
}

//------------------------------------------------------------------------------
/**
	Return the height at a given grid node, for a chosen water algorithm.
	The values must be already clamped to 0-SampleSize which is done by nWaterNode

	@param		x		X position in base water grid
	@param		z		Z position in base water grid
	@param		algo	The water algorithm to be queried @see SetType

	@return				Y position in base water grid at passed position
*/
float nWaterServer::GetHeight(int x, int z, int algo)
{
	switch(algo)
	{
	case N_STATISTIC: 
	case N_CHOPPY: return m_Choppy != NULL ? 
					   (getCMap())[x+(z*SampleSize)].real()*0.04f 
					   : 0.0f;
	case N_GERSTNER: return this->m_Gerstner != NULL ?
						(this->m_Gerstner->getVertices())[x+(z*SampleSize)].y *0.04f
						: 0.0f;
	default: return 0.0f;
	}
}

//------------------------------------------------------------------------------
/**
	Return the height at a given grid node, for a chosen water algorithm.
	The values must be already clamped to 0-SampleSize which is done by nWaterNode

	@param		x		X position in base water grid
	@param		z		Z position in base water grid
	@param		algo	The water algorithm to be queried @see SetType

	@return				Y position in base water grid at passed position
*/
float nWaterServer::GetHeight(float x, float z, int algo)
{
	switch(algo)
	{
	case N_STATISTIC: 
		return m_Choppy != NULL ? GetSH(x,z) : 0.0f;
	case N_CHOPPY: return m_Choppy != NULL ? GetCH(x,z) : 0.0f;
	case N_GERSTNER: return m_Gerstner != NULL ? GetGH(x,z) : 0.0f;
	default: return 0.0f;
	}
}

//------------------------------------------------------------------------------
/**
	Return the height at a given grid node, for statistical water.
	The values must be already clamped to 0-SampleSize which is done by nWaterNode

	@param		x		X position in statistical water grid
	@param		z		Z position in statistical water grid

	@todo test correct working of position decision

	@return				Y position in statistical water grid at passed position
*/
float nWaterServer::GetSH(float x, float z)
{
	int _x=x;
	int _z=z;
n_printf("_x: %d _z: %d",_x, _z);
	if(z<=0.0f || x < z)
	{
		//lower half of quad
		float ya = (getCMap())[_x+(_z*SampleSize)].real();
		float yb = (getCMap())[_x+1+(_z*SampleSize)].real();
		float yc = (getCMap())[_x+1+((_z+1)*SampleSize)].real();
n_printf(" %f =(%f-%d)*(%f-%f) + (%f-%d)*(%f-%f) *0.04f \n",(( (z-_z)*(yb-ya) + (x-_x)*(yc-yb) ) * 0.04f),z,_z,yb,ya,x,_x,yc,yb );
		return ( (z-_z)*(yb-ya) + (x-_x)*(yc-yb) ) * 0.04f;
	}
//	else if(x/z <= 1.0f)
//	{
//		//lower half of quad
//		float ya = (getCMap())[_x+(_z*SampleSize)].real();
//		float yb = (getCMap())[_x+1+(_z*SampleSize)].real();
//		float yc = (getCMap())[_x+1+((_z+1)*SampleSize)].real();
//n_printf(" %f =(%f-%d)*(%f-%f) + (%f-%d)*(%f-%f) *0.04f \n",(( (z-_z)*(yb-ya) + (x-_x)*(yc-yb) ) * 0.04f),z,_z,yb,ya,x,_x,yc,yb );
//		return ( (z-_z)*(yb-ya) + (x-_x)*(yc-yb) ) * 0.04f;
//
//	}
	else
	{
		//upper half of quad
		float ya = (getCMap())[_x+(_z*SampleSize)].real();
		float yd = (getCMap())[_x+((_z+1)*SampleSize)].real();
		float yc = (getCMap())[_x+1+((_z+1)*SampleSize)].real();
n_printf(" %f =(%f-%d)*(%f-%f) + (%f-%d)*(%f-%f) *0.04f \n",(( (z-_z)*(yd-ya) + (x-_x)*(yc-yd) ) * 0.04f),z,_z,yd,ya,x,_x,yc,yd );
		return ( (z-_z)*(yd-ya) + (x-_x)*(yc-yd) ) * 0.04f;
	}
}

//------------------------------------------------------------------------------
/**
	Return the height at a given grid node, for choppy water.
	The values must be already clamped to 0-SampleSize which is done by nWaterNode.
	The choppy height is not an easy case for height decisions, the current algorithm
	works as the statistical water but also includes the X and Z deviations. The
	deviations of grid nodes is the critical point as it can occur that a grid node
	has shifted to a position where it is no longer part of the triangle residing
	at the real height query position.


      H0----H1----H2
    /     /      /
	    /      /
	  /      /
	/      /
	X1----X2

	As you can see above the height value at world position lying between X1 and X2 is
	due to the deviation to be computed based on the nodes X1 and X0 (not shown here) to
	get the correct height.

	@param		x		X position in statistical water grid
	@param		z		Z position in statistical water grid

	@todo not clear at the moment how this should be made to work correctly and fast

	@return				Y position in statistical water grid at passed position
*/
//choppy height no deviation in x and z most simple case of all
float nWaterServer::GetCH(float x, float z)
{
	int _x=x;
	int _z=z;
	//float xd = getXMap()[_x + (-z * SampleSize)].real();
	//float yd = getZMap()[_x + (-z * SampleSize)].real();
	//xmap[a + (b * SampleSize)].real()*0.4f;
n_printf("_x: %d _z: %d",_x, _z);
	if(z<=0.0f || x < z)
	{
		//lower half of quad
		float ya = (getCMap())[_x+(_z*SampleSize)].real();
		float yb = (getCMap())[_x+1+(_z*SampleSize)].real();
		float yc = (getCMap())[_x+1+((_z+1)*SampleSize)].real();
n_printf(" %f =(%f-%d)*(%f-%f) + (%f-%d)*(%f-%f) *0.04f \n",(( (z-_z)*(yb-ya) + (x-_x)*(yc-yb) ) * 0.04f),z,_z,yb,ya,x,_x,yc,yb );
		return ( (z-_z + (getZMap())[_x+(_z*SampleSize)].real())*(yb-ya) + 
				 (x-_x + (getXMap())[_x+(_z*SampleSize)].real())*(yc-yb) ) * 0.04f;
	}
//	else if(x/z <= 1.0f)
//	{
//		//lower half of quad
//		float ya = (getCMap())[_x+(_z*SampleSize)].real();
//		float yb = (getCMap())[_x+1+(_z*SampleSize)].real();
//		float yc = (getCMap())[_x+1+((_z+1)*SampleSize)].real();
//n_printf(" %f =(%f-%d)*(%f-%f) + (%f-%d)*(%f-%f) *0.04f \n",(( (z-_z)*(yb-ya) + (x-_x)*(yc-yb) ) * 0.04f),z,_z,yb,ya,x,_x,yc,yb );
//		return ( (z-_z)*(yb-ya) + (x-_x)*(yc-yb) ) * 0.04f;
//
//	}
	else
	{
		//upper half of quad
		float ya = (getCMap())[_x+(_z*SampleSize)].real();
		float yd = (getCMap())[_x+((_z+1)*SampleSize)].real();
		float yc = (getCMap())[_x+1+((_z+1)*SampleSize)].real();
n_printf(" %f =(%f-%d)*(%f-%f) + (%f-%d)*(%f-%f) *0.04f \n",(( (z-_z)*(yd-ya) + (x-_x)*(yc-yd) ) * 0.04f),z,_z,yd,ya,x,_x,yc,yd );
		return ( (z-_z + (getCMap())[_x+(_z*SampleSize)].real())*(yd-ya) + 
			     (x-_x + (getCMap())[_x+(_z*SampleSize)].real())*(yc-yd) ) * 0.04f;
	}
}


//------------------------------------------------------------------------------
/**
	Return the height at a given grid node, for gerstner water.
	The values must be already clamped to 0-SampleSize which is done by nWaterNode.
	The current algorithm suffers from the same problem as the choppy water algorithm

	@param		x		X position in gerstner water grid
	@param		z		Z position in gerstner water grid

	@todo resolve shift problem as is with choppy water too.

	@return				Y position in gerstner water grid at passed position
*/
float nWaterServer::GetGH(float x, float z)
{
	int _x=x;
	int _z=z;
n_printf("_x: %d _z: %d",_x, _z);
	if(z<=0.0f || x < z)
	{
		//lower half of quad
		float ya = (getGMap())[_x+(_z*SampleSize)].y;
		float yb = (getGMap())[_x+1+(_z*SampleSize)].y;
		float yc = (getGMap())[_x+1+((_z+1)*SampleSize)].y;
n_printf(" %f =(%f-%d)*(%f-%f) + (%f-%d)*(%f-%f) *0.04f \n",
		 (( (z-_z + (getGMap())[_x+(_z*SampleSize)].z)*(yb-ya) + 
			     (x-_x + (getGMap())[_x+(_z*SampleSize)].x)*(yc-yb) ) * 0.04f),
				 z,_z,yb,ya,x,_x,yc,yb );
		return ( (z-_z + (getGMap())[_x+(_z*SampleSize)].z)*(yb-ya) + 
			     (x-_x + (getGMap())[_x+(_z*SampleSize)].x)*(yc-yb) ) * 0.04f;
	}
//	else if(x/z <= 1.0f)
//	{
//		//lower half of quad
//		float ya = (getCMap())[_x+(_z*SampleSize)].real();
//		float yb = (getCMap())[_x+1+(_z*SampleSize)].real();
//		float yc = (getCMap())[_x+1+((_z+1)*SampleSize)].real();
//n_printf(" %f =(%f-%d)*(%f-%f) + (%f-%d)*(%f-%f) *0.04f \n",(( (z-_z)*(yb-ya) + (x-_x)*(yc-yb) ) * 0.04f),z,_z,yb,ya,x,_x,yc,yb );
//		return ( (z-_z)*(yb-ya) + (x-_x)*(yc-yb) ) * 0.04f;
//
//	}
	else
	{
		//upper half of quad
		float ya = getGMap()[_x+(_z*SampleSize)].y;
		float yd = getGMap()[_x+((_z+1)*SampleSize)].y;
		float yc = getGMap()[_x+1+((_z+1)*SampleSize)].y;
n_printf(" %f =(%f-%d)*(%f-%f) + (%f-%d)*(%f-%f) *0.04f \n",
		 (( (z-_z + (getGMap())[_x+(_z*SampleSize)].z)*(yd-ya) + 
			     (x-_x + (getGMap())[_x+(_z*SampleSize)].x)*(yc-yd) ) * 0.04f),
				 z,_z,yd,ya,x,_x,yc,yd );

		return ( (z-_z + (getGMap())[_x+(_z*SampleSize)].z)*(yd-ya) + 
			     (x-_x + (getGMap())[_x+(_z*SampleSize)].x)*(yc-yd) ) * 0.04f;
	}
}
